;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.netty.http2.server
  (:require [utilis.map :refer [map-keys]]
            [byte-streams.core :as bs]
            [clojure.string :as st])
  (:import [io.netty.buffer ByteBuf]
           [io.netty.channel
            ChannelPromise
            ChannelDuplexHandler
            ChannelPipeline
            ChannelHandler
            ChannelHandlerContext]
           [java.util.concurrent ExecutorService]
           [io.netty.util ReferenceCounted]
           [io.netty.util.concurrent GenericFutureListener]
           [io.netty.handler.codec.http.websocketx
            TextWebSocketFrame
            CloseWebSocketFrame
            WebSocket13FrameDecoder
            WebSocket13FrameEncoder
            WebSocketDecoderConfig]
           [io.netty.handler.codec.http.websocketx.extensions
            WebSocketExtensionDecoder
            WebSocketExtensionEncoder]
           [io.netty.handler.codec.http2
            DefaultHttp2DataFrame
            DefaultHttp2Headers
            DefaultHttp2HeadersFrame
            DefaultHttp2PingFrame
            DefaultHttp2WindowUpdateFrame
            Http2ResetFrame
            Http2PingFrame
            Http2WindowUpdateFrame
            Http2FrameCodecBuilder
            Http2HeadersFrame
            Http2DataFrame
            Http2FrameStream
            Http2Headers
            Http2Settings]))

(def ByteArray (type (byte-array [])))

(def SETTINGS_ENABLE_CONNECT_PROTOCOL (char 8))
(def SETTINGS_INITIAL_WINDOW_SIZE (char 4))

(def DEFAULT_INITIAL_WINDOW_SIZE (long (dec (Math/pow 2 16))))

(defn websocket-request?
  [req]
  (boolean (and (= "websocket" (get-in req [:headers ":protocol"]))
                (= "13" (get-in req [:headers "sec-websocket-version"])))))

(defn add-all-headers
  [^Http2Headers headers req headers-map]
  (doseq [[k v] (map-keys #(st/lower-case
                            (str (if (keyword? %)
                                   (name %)
                                   %)))
                          headers-map)]
    (.add headers k (str v)))
  headers)

(defn headers
  [^Http2HeadersFrame msg]
  (->> (.headers msg)
       .iterator
       iterator-seq
       (map (fn [[k v]]
              [(str k) (str v)]))
       (into {})))

(defn request
  [^Http2HeadersFrame msg]
  (let [headers (headers msg)
        path (get headers ":path")
        [uri query-string] (st/split path #"\?")]
    {:headers headers
     :protocol-version "http/2"
     :uri uri
     :path path
     :scheme (keyword (get headers ":scheme"))
     :query-string query-string
     :request-method (-> headers
                         (get ":method")
                         st/lower-case
                         keyword)
     :query-params (when query-string
                     (->> (st/split query-string #"&")
                          (map #(st/split % #"\="))
                          (into {})))}))

(defn handle-http2-headers-frame
  [^ChannelHandlerContext ctx ^Http2HeadersFrame headers-frame ^ExecutorService exec-service handler]
  (let [websocket? (.contains (.headers headers-frame) ":protocol" "websocket" true)]
    (when (or websocket? (.isEndStream headers-frame))
      (let [req (request headers-frame)]
        (.submit exec-service
                 (reify Runnable
                   (run [_]
                     (try
                       (.fireUserEventTriggered
                        ctx {:req req
                             :res (handler req)
                             :headers-frame headers-frame})
                       (catch Exception e
                         (.fireUserEventTriggered
                          ctx {:req req
                               :res {:status 500
                                     :body "Internal Error Occurred"
                                     :headers {}}
                               :headers-frame headers-frame})
                         (throw e))))))
        nil))))

(defn safe-remove
  [^ChannelPipeline pipeline ^String handler-name]
  (when ((set (.names pipeline)) handler-name)
    (.remove pipeline handler-name)))

(defn configure-websocket-pipeline
  [^ChannelPipeline pipeline
   {:keys [^WebSocket13FrameDecoder frame-decoder
           ^WebSocket13FrameEncoder frame-encoder
           ^WebSocketExtensionDecoder compression-decoder
           ^WebSocketExtensionEncoder compression-encoder
           ^ExecutorService exec-service
           ^Http2FrameStream http2-stream
           handlers]}]
  (let [{:keys [on-text-message on-close]} handlers
        run-exec (fn [f]
                   (.submit exec-service
                            (reify Runnable
                              (run [_]
                                (f)))))
        websocket-handler (proxy [ChannelDuplexHandler] []
                            (channelRead [^ChannelHandlerContext ctx msg]
                              (try (cond
                                     (instance? TextWebSocketFrame msg)
                                     (let [text (.text ^TextWebSocketFrame msg)]
                                       (run-exec #(on-text-message text)))

                                     (instance? CloseWebSocketFrame msg)
                                     (run-exec #(on-close)))
                                   (finally
                                     (when (instance? ReferenceCounted msg)
                                       (.release ^ReferenceCounted msg))
                                     (.fireChannelReadComplete ctx)))))

        cleanup (atom nil)
        handlers [["0" (proxy [ChannelDuplexHandler] []
                         (write [^ChannelHandlerContext ctx msg ^ChannelPromise p]
                           (if (instance? ByteBuf msg)
                             (let [frame (DefaultHttp2DataFrame. msg false)]
                               (.writeAndFlush ctx (.stream frame http2-stream))
                               (.release frame))
                             (.writeAndFlush ctx msg))))]
                  ["1" (proxy [ChannelDuplexHandler] []
                         (channelRead [^ChannelHandlerContext ctx msg]
                           (let [data-frame? (instance? Http2DataFrame msg)]
                             (try (cond
                                    (instance? Http2DataFrame msg)
                                    (let [msg ^Http2DataFrame msg
                                          ^ByteBuf content (.content msg)]
                                      (.fireChannelRead ctx content)
                                      (->> (-> (.initialFlowControlledBytes msg)
                                               (DefaultHttp2WindowUpdateFrame.)
                                               (.stream http2-stream))
                                           (.writeAndFlush ctx)))

                                    (instance? Http2PingFrame msg)
                                    (.writeAndFlush ctx (DefaultHttp2PingFrame. 1 true))

                                    (instance? Http2ResetFrame msg)
                                    (@cleanup)

                                    (instance? Http2WindowUpdateFrame msg) nil

                                    :else nil)
                                  (finally
                                    (when (and (instance? ReferenceCounted msg)
                                               (not (instance? Http2DataFrame msg)))
                                      (.release ^ReferenceCounted msg))
                                    (when (not data-frame?)
                                      (.fireChannelReadComplete ctx)))))))]
                  ["2" frame-decoder]
                  ["3" compression-decoder]
                  ["4" frame-encoder]
                  ["5" compression-encoder]
                  ["6" websocket-handler]
                  ["7" (proxy [ChannelDuplexHandler] []
                         (write [^ChannelHandlerContext ctx msg ^ChannelPromise p]
                           (when (instance? TextWebSocketFrame msg)
                             (.write ctx msg p))))]
                  ["8" (proxy [ChannelDuplexHandler] []
                         (userEventTriggered [^ChannelHandlerContext ctx event]
                           (when (and (map? event) (:send event))
                             (-> (.channel ctx)
                                 (.writeAndFlush (TextWebSocketFrame. (:send event)))
                                 (.addListener (reify GenericFutureListener
                                                 (operationComplete [_ _]
                                                   )))))))]]]


    (reset! cleanup
            #(doseq [[name _] handlers]
               (safe-remove pipeline name)))

    (doseq [[name handler] handlers]
      (.addLast pipeline name handler))))

(defn configure-websocket
  [^ChannelHandlerContext ctx ^Http2HeadersFrame msg req {:keys [headers] :as res} ^ExecutorService exec-service]
  (when (not (get (set (.names (.pipeline ctx))) "websocket-handler"))
    (let [decoder-config (-> (WebSocketDecoderConfig/newBuilder)
                             (.allowExtensions true)
                             (.build))
          {:keys [compression-extension handlers]} (meta res)
          compression-encoder (when compression-extension (.newExtensionEncoder compression-extension))
          compression-decoder (when compression-extension (.newExtensionDecoder compression-extension))
          headers (doto (DefaultHttp2Headers.)
                    (.status (str 200))
                    (add-all-headers req (or (not-empty headers) {})))
          frame-decoder (WebSocket13FrameDecoder. decoder-config)
          frame-encoder (WebSocket13FrameEncoder. false)
          http2-stream (.stream msg)
          {:keys [on-open]} handlers]
      (->> http2-stream
           (.stream (DefaultHttp2HeadersFrame. headers))
           (.writeAndFlush ctx))
      (configure-websocket-pipeline
       (.pipeline ctx)
       {:frame-decoder frame-decoder
        :frame-encoder frame-encoder
        :compression-decoder compression-decoder
        :compression-encoder compression-encoder
        :handlers handlers
        :exec-service exec-service
        :http2-stream http2-stream})
      (on-open {:send #(.fireUserEventTriggered ctx {:send %})
                :close (fn [] (.close (.channel ctx)))})
      nil)))

(defn http2-response-handler
  ^ChannelHandler
  [handler ^ExecutorService exec-service]
  (proxy [ChannelDuplexHandler] []
    (userEventTriggered [^ChannelHandlerContext ctx event]
      (when (and (map? event) (:req event) (:res event) (:headers-frame event))
        (let [{:keys [res req headers-frame]} event
              headers-frame ^Http2HeadersFrame headers-frame
              {:keys [status body headers]} res]
          (if (and (websocket-request? req)
                   (= 200 (:status res)))
            (configure-websocket ctx headers-frame req res exec-service)
            (let [headers (doto (DefaultHttp2Headers.)
                            (.status (str status))
                            (add-all-headers req headers))]
              (->> (.stream headers-frame)
                   (.stream (DefaultHttp2HeadersFrame. headers))
                   (.write ctx))
              (when body
                (let [^ByteBuf content (-> ctx .alloc .buffer)
                      body-bytes (bs/convert body ByteArray)]
                  (.writeBytes content body-bytes)
                  (when (instance? java.io.Closeable body)
                    (.close ^java.io.Closeable body))
                  (->> (.stream headers-frame)
                       (.stream (DefaultHttp2DataFrame. content true))
                       (.write ctx))))
              (.flush ctx))))))))

(defn ^ChannelHandler http2-request-handler
  [^ExecutorService exec-service handler]
  (proxy [ChannelDuplexHandler] []
    (channelRead [^ChannelHandlerContext ctx msg]
      (if (instance? Http2HeadersFrame msg)
        (handle-http2-headers-frame ctx ^Http2HeadersFrame msg exec-service handler)
        (proxy-super channelRead ctx msg)))))

(defn configure-http2-request-handler
  [^ChannelPipeline pipeline ^ExecutorService exec-service handler]
  (doto pipeline
    (.addLast "request-handler" (http2-request-handler exec-service handler))
    (.addLast "response-handler" (http2-response-handler handler exec-service))))

(defn configure-http2-frame-codec-builder
  [^ChannelPipeline pipeline]
  (doto pipeline
    (.addLast "frame-codec"
              (-> (Http2FrameCodecBuilder/forServer)
                  (.initialSettings
                   (doto (Http2Settings/defaultSettings)
                     (.put SETTINGS_ENABLE_CONNECT_PROTOCOL (long 1))
                     (.put SETTINGS_INITIAL_WINDOW_SIZE (long DEFAULT_INITIAL_WINDOW_SIZE))))
                  .build))))

(defn configure-http2-pipeline
  [^ChannelPipeline pipeline ^ExecutorService exec-service handler]
  (doto pipeline
    (configure-http2-frame-codec-builder)
    (configure-http2-request-handler exec-service handler)))
