(ns via.netty.http2.client
  (:require [utilis.map :refer [map-vals]])
  (:import [java.net.http HttpClient HttpRequest HttpClient$Version HttpResponse$BodyHandlers]
           [java.util.concurrent LinkedBlockingQueue Executors]
           [java.net URI]))

(def default-http-client-executor
  (delay
    (Executors/newFixedThreadPool
     (* 2 (.availableProcessors (Runtime/getRuntime))))))

(defn new-http-client
  []
  (-> (HttpClient/newBuilder)
      (.version HttpClient$Version/HTTP_2)
      (.executor @default-http-client-executor)
      (.build)))

(def default-http-client
  (atom (delay (new-http-client))))

(defn http-client
  [client]
  (or client @@default-http-client))

(defn reset-default-client!
  []
  (reset! default-http-client (delay (new-http-client))))

(defn request
  [{:keys [url headers method client connection-reset-retries]
    :or {method :get
         connection-reset-retries 1}
    :as args}]
  (try (let [^HttpClient client (http-client client)
             request (cond-> (HttpRequest/newBuilder)
                       true (.uri (URI/create url))
                       true (as-> %
                                (condp = method
                                  :get (.GET %)
                                  :post (.POST %)
                                  :put (.PUT %)
                                  :options (.OPTIONS %)))
                       (seq headers) (.headers (into-array (mapcat identity headers)))
                       true (.build))
             response (.send client request (HttpResponse$BodyHandlers/ofByteArray))]
         {:status (.statusCode response)
          :body (.body response)
          :headers (-> (->> (.headers response)
                            .map
                            (into {})
                            (map-vals first))
                       (dissoc ":status"))})
       (catch java.io.IOException e
         (if (and (= (.getMessage e) "Connection reset")
                  (pos? connection-reset-retries))
           (do (reset-default-client!)
               (request (assoc args :connection-reset-retries (dec connection-reset-retries))))
           (throw e)))))
