(ns via.netty.http.websocket
  (:require [utilis.map :refer [map-keys map-vals compact]]
            [byte-streams.core :as bs]
            [manifold.deferred :as d]
            [manifold.executor :as e]
            [manifold.stream :as s]
            [manifold.stream.core :as manifold]
            [clojure.string :as st])
  (:import [java.io IOException]
           [io.netty.bootstrap Bootstrap ServerBootstrap]
           [io.netty.buffer ByteBuf Unpooled]
           [io.netty.channel
            ChannelFutureListener
            ChannelPromise
            ChannelDuplexHandler
            Channel ChannelFuture ChannelOption
            ChannelPipeline EventLoopGroup
            DefaultChannelPipeline
            ChannelHandler FileRegion
            ChannelInboundHandler
            ChannelOutboundHandler
            ChannelHandlerContext
            ChannelInitializer
            SimpleChannelInboundHandler]
           [java.util.concurrent
            ConcurrentHashMap
            CancellationException
            ScheduledFuture
            TimeUnit
            ThreadFactory
            Executors
            ExecutorService]
           [io.netty.channel.epoll Epoll EpollEventLoopGroup
            EpollServerSocketChannel
            EpollSocketChannel
            EpollDatagramChannel]
           [io.netty.util Attribute AttributeKey]
           [io.netty.handler.codec Headers]
           [io.netty.handler.codec.http
            DefaultFullHttpResponse
            DefaultHttpHeaders
            FullHttpResponse
            HttpVersion
            HttpResponseStatus
            HttpRequest
            HttpClientCodec
            HttpServerCodec
            HttpObjectAggregator]
           [io.netty.handler.codec.http.websocketx
            WebSocketVersion
            WebSocketServerHandshakerFactory
            WebSocketClientHandshakerFactory
            WebSocketClientHandshaker
            TextWebSocketFrame
            CloseWebSocketFrame
            WebSocketFrame
            WebSocket13FrameDecoder
            WebSocket13FrameEncoder
            WebSocketDecoderConfig]
           [io.netty.handler.codec.http.websocketx.extensions
            WebSocketServerExtension
            WebSocketExtensionData
            WebSocketExtensionDecoder
            WebSocketExtensionEncoder]
           [io.netty.handler.codec.http.websocketx.extensions.compression
            WebSocketClientCompressionHandler
            PerMessageDeflateServerExtensionHandshaker]
           [io.netty.handler.codec.http2
            DefaultHttp2DataFrame
            DefaultHttp2Headers
            DefaultHttp2HeadersFrame
            EmptyHttp2Headers
            Http2FrameCodecBuilder
            Http2HeadersFrame
            Http2DataFrame
            Http2FrameStream
            Http2Headers
            Http2Settings]
           [io.netty.channel.nio NioEventLoopGroup]
           [io.netty.channel.socket
            ServerSocketChannel
            SocketChannel]
           [io.netty.channel.socket.nio
            NioServerSocketChannel
            NioSocketChannel
            NioDatagramChannel]
           [io.netty.handler.ssl
            SslContext
            SslContextBuilder
            SslHandler
            SslHandshakeCompletionEvent
            ApplicationProtocolNegotiationHandler
            ApplicationProtocolNames]
           [io.netty.handler.ssl.util
            InsecureTrustManagerFactory]
           [java.net URI SocketAddress InetSocketAddress]
           [java.nio ByteBuffer]))

(defn websocket-client-handler
  [^WebSocketClientHandshaker handshaker]
  (let [handshake-future (atom nil)]
    (proxy [SimpleChannelInboundHandler] []
      (handlerAdded [^ChannelHandlerContext ctx]
        (prn :foo1)
        (reset! handshake-future (.newPromise ctx)))
      (channelActive [^ChannelHandlerContext ctx]
        (prn :foo2)
        (.handshake handshaker (.channel ctx)))
      (channelInactive [^ChannelHandlerContext ctx]
        (prn :foo3)
        ;; disconnected
        )
      (channelRead [^ChannelHandlerContext ctx msg]
        (prn :foo4.0)
        (let [ch (.channel ctx)]
          (prn :foo4)
          (cond
            (not (.isHandshakeComplete ctx))
            (try (prn :foo5)
                 (.finishHandshake handshaker ch ^FullHttpResponse msg)
                 ;; connected
                 (.setSuccess @handshake-future)
                 (catch Exception e
                   ;; handshake failed
                   (.setFailure @handshake-future)))

            (instance? msg FullHttpResponse)
            (throw (ex-info "Unexpected FullHttpResponse" {:msg msg}))

            :else
            (let [^WebSocketFrame frame msg]
              (prn :frame msg))))))))

(defn connect
  [uri]
  (let [group (NioEventLoopGroup.)
        b (Bootstrap.)
        uri (URI/create uri)
        scheme (or (.getScheme uri) "ws")
        host (or (.getHost uri) "127.0.0.1")
        port (if (= -1 (.getPort uri))
               (condp = scheme
                 "ws" 80
                 "wss" 443
                 -1)
               (.getPort uri))
        _ (when (not (#{"ws" "wss"} scheme))
            (throw (ex-info "Unsupported scheme" {:scheme scheme})))
        ssl? (= "wss" scheme)
        ssl-ctx (when ssl?
                  (-> (SslContextBuilder/forClient)
                      (.trustManager InsecureTrustManagerFactory/INSTANCE)
                      .build))
        handler (websocket-client-handler
                 (WebSocketClientHandshakerFactory/newHandshaker
                  uri
                  WebSocketVersion/V13
                  nil
                  true
                  (DefaultHttpHeaders.)))]
    (-> b
        (.group group)
        (.channel ^Class NioServerSocketChannel)
        (.handler (proxy [ChannelInitializer] []
                    (initChannel [^SocketChannel ch]
                      (let [p (.pipeline ch)]
                        (when ssl-ctx
                          (.addLast p "ssl-handler" (.newHandler ssl-ctx (.alloc ch) host port)))
                        (doto p
                          (.addLast "http-codec" (HttpClientCodec.))
                          (.addLast "aggregator" (HttpObjectAggregator. 8192))
                          (.addLast "compression-handler" WebSocketClientCompressionHandler/INSTANCE)
                          (.addLast "ws-handler" handler)))))))

    (let [ch (-> b
                 (.connect (.getHost uri) port)
                 .sync)]
      (prn :connection ch)
      (fn []
        (.sync (.closeFuture ch))))))

(comment

  (def stop-connection
    (connect "wss://localhost:3449/via"))

  )
