;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.netty.http2.client.websocket
  (:require [via.defaults :refer [default-via-endpoint]])
  (:import [java.util HashMap]
           [java.util.concurrent TimeUnit]

           [io.netty.bootstrap Bootstrap]
           [io.netty.buffer ByteBuf]
           [io.netty.channel.socket SocketChannel]
           [io.netty.channel
            Channel
            ChannelInitializer
            ChannelDuplexHandler
            ChannelHandler
            ChannelHandlerContext
            ChannelInboundHandlerAdapter
            ChannelOption
            ChannelPromise
            EventLoopGroup]
           [io.netty.channel.nio NioEventLoopGroup]
           [io.netty.channel.socket.nio NioSocketChannel]
           [io.netty.handler.codec.http.websocketx
            TextWebSocketFrame
            WebSocket13FrameDecoder
            WebSocket13FrameEncoder
            WebSocketDecoderConfig]
           [io.netty.handler.codec.http.websocketx.extensions
            WebSocketClientExtension
            WebSocketExtensionData
            WebSocketExtensionDecoder
            WebSocketExtensionEncoder]
           [io.netty.handler.codec.http.websocketx.extensions.compression
            PerMessageDeflateClientExtensionHandshaker]
           [io.netty.handler.codec.http2
            Http2FrameCodecBuilder
            Http2MultiplexHandler
            Http2SettingsFrame
            Http2SettingsAckFrame
            DefaultHttp2DataFrame
            DefaultHttp2Headers
            DefaultHttp2HeadersFrame
            Http2DataFrame
            Http2Headers
            Http2HeadersFrame
            Http2StreamChannelBootstrap
            Http2SecurityUtil]
           [io.netty.handler.ssl
            SslContextBuilder
            SslContext
            SslProvider
            SupportedCipherSuiteFilter
            ApplicationProtocolConfig
            ApplicationProtocolConfig$Protocol
            ApplicationProtocolConfig$SelectorFailureBehavior
            ApplicationProtocolConfig$SelectedListenerFailureBehavior
            ApplicationProtocolNegotiationHandler
            ApplicationProtocolNames]
           [io.netty.handler.ssl.util
            InsecureTrustManagerFactory]))

(defn http2-stream-channel-initializer
  ^ChannelInitializer []
  (proxy [ChannelInitializer] []
    (initChannel [^Channel ch]
      )))

(defn http2-frame-duplex-handler
  ^ChannelHandler [^ChannelPromise settings-promise]
  (let [got-settings (atom false)
        got-settings-ack (atom false)]
    (proxy [ChannelDuplexHandler] []
      (channelRead [^ChannelHandlerContext ctx msg]
        (prn :duplex msg)
        (when (or (not @got-settings)
                  (not @got-settings-ack))
          (cond
            (instance? Http2SettingsFrame msg)
            (reset! got-settings true)

            (instance? Http2SettingsAckFrame msg)
            (reset! got-settings-ack true))
          (when (and @got-settings
                     @got-settings-ack)
            (.setSuccess settings-promise)))))))

(defn pass-through
  ^ChannelDuplexHandler [k]
  (proxy [ChannelDuplexHandler] []
    (channelRead [^ChannelHandlerContext ctx msg]
      (locking Object
        (prn :pass-through k msg))
      (let [^ChannelDuplexHandler this this]
        (proxy-super channelRead ctx msg)))))

(defn apn-handler
  ^ApplicationProtocolNegotiationHandler [settings-promise]
  (proxy [ApplicationProtocolNegotiationHandler] [ApplicationProtocolNames/HTTP_2]
    (configurePipeline [^ChannelHandlerContext ctx ^String protocol]
      (if (= ApplicationProtocolNames/HTTP_2 protocol)
        (let [p (.pipeline ctx)
              frame-codec (.build (Http2FrameCodecBuilder/forClient))
              multiplex-handler (Http2MultiplexHandler. (http2-stream-channel-initializer))
              ^ChannelHandler frame-duplex-handler (http2-frame-duplex-handler settings-promise)]
          (.addLast p "foobar1" ^ChannelDuplexHandler (pass-through 1))
          (.addLast p "frame-codec" frame-codec)
          (.addLast p "foobar2" ^ChannelDuplexHandler (pass-through 2))
          (.addLast p "multiplex-handler" multiplex-handler)
          (.addLast p "foobar3" ^ChannelDuplexHandler (pass-through 3))
          (.addLast p "frame-duplex-handler" frame-duplex-handler)
          (.addLast p "foobar4" ^ChannelDuplexHandler (pass-through 4)))
        (do (.close ctx)
            (throw (ex-info "Protocol not supported" {:protocol protocol})))))))

(defn client-initializer
  ^ChannelInitializer [{:keys [ssl-context host port settings-promise]}]
  (proxy [ChannelInitializer] []
    (initChannel [^SocketChannel ch]
      (let [settings-promise (reset! settings-promise (.newPromise ch))
            pipeline (.pipeline ch)]
        (->> (int port)
             (.newHandler ^SslContext ssl-context (.alloc ch) ^String host)
             (.addLast pipeline "ssl-handler"))
        (.addLast pipeline "apn-handler" (apn-handler settings-promise))))))

(defn websocket-request-headers-frame
  ^Http2HeadersFrame [{:keys [host port path]}]
  (let [^String host host
        port (int port)
        ^String path path
        headers (DefaultHttp2Headers.)
        host-header (str host ":" port)]
    (.method headers "GET")
    (.add headers "host" host-header)
    (.add headers "accept-encoding" "gzip, deflate, br")
    (.add headers "sec-websocket-extensions" "permessage-deflate; client_max_window_bits")
    (.add headers "sec-websocket-version" "13")
    (.authority headers host-header)
    (.path headers path)
    (.scheme headers "https")
    (.add headers ":protocol" "websocket")
    (DefaultHttp2HeadersFrame. headers false)))

(defn stream-channel-handler
  ^ChannelHandler [{:keys [host port path max-frame-size send-message]
                    :or {max-frame-size (Math/pow 2 16)}}]
  (proxy [ChannelInboundHandlerAdapter] []
    (channelActive [^ChannelHandlerContext ctx]
      (let [^Http2HeadersFrame handshake-frame (websocket-request-headers-frame
                                                {:host host
                                                 :port port
                                                 :path path})]
        (.writeAndFlush ctx handshake-frame)
        (.fireChannelActive ctx)))
    (channelRead [^ChannelHandlerContext ctx msg]
      (when (instance? Http2HeadersFrame msg)
        (let [^Http2HeadersFrame headers-frame msg
              headers (.headers headers-frame)]
          (when (and (= "200" (-> headers .status str))
                     (.contains headers "sec-websocket-extensions"))
            (let [decoder-config (-> (WebSocketDecoderConfig/newBuilder)
                                     (.allowExtensions true)
                                     (.maxFramePayloadLength max-frame-size)
                                     .build)
                  frame-decoder (WebSocket13FrameDecoder. decoder-config)
                  frame-encoder (WebSocket13FrameEncoder. true)
                  compression-handshaker (PerMessageDeflateClientExtensionHandshaker.)
                  extension-data (WebSocketExtensionData. "permessage-deflate" {})
                  compression-extension (.handshakeExtension compression-handshaker extension-data)
                  compression-encoder (.newExtensionEncoder compression-extension)
                  compression-decoder (.newExtensionDecoder compression-extension)
                  p (.pipeline ctx)
                  data-frame-writer (proxy [ChannelDuplexHandler] []
                                      (write [^ChannelHandlerContext ctx msg ^ChannelPromise promise]
                                        (let [data-frame (-> ^ByteBuf msg
                                                             (DefaultHttp2DataFrame. false)
                                                             (.stream (.stream headers-frame)))]
                                          (.write ctx data-frame))))]

              (.addLast p "frame-inbound-handler"
                        (proxy [ChannelDuplexHandler] []
                          (channelRead [^ChannelHandlerContext ctx ^Http2DataFrame msg]
                            #_(n/acquire msg)
                            (let [^ChannelDuplexHandler this this]
                              (prn :inbound msg)
                              (proxy-super channelRead ctx (.content msg))))))
              (.addLast p "data-frame-writer" data-frame-writer)
              (.addLast p "frame-decoder" frame-decoder)
              (.addLast p "compression-decoder" compression-decoder)
              (.addLast p "frame-encoder" frame-encoder)
              (.addLast p "compression-encoder" compression-encoder)

              (reset! send-message
                      (fn [^String message]
                        (.writeAndFlush p (TextWebSocketFrame. message)))))))))))

(defn websocket-client
  ([] (websocket-client nil))
  ([{:keys [host port ssl-context path]
     :or {host "localhost"
          port 3449
          path default-via-endpoint}}]
   (let [worker-group (NioEventLoopGroup.)
         settings-promise (atom nil)
         ^ChannelInitializer initializer (client-initializer
                                          {:ssl-context ssl-context
                                           :host host
                                           :port port
                                           :settings-promise settings-promise})
         b (doto (Bootstrap.)
             (.group worker-group)
             (.channel ^NioSocketChannel NioSocketChannel)
             (.option ChannelOption/SO_KEEPALIVE true)
             (.remoteAddress ^String host ^long port)
             (.handler initializer))
         channel (-> b
                     .connect
                     .syncUninterruptibly
                     .channel)
         ^ChannelPromise settings-promise @settings-promise]
     (.await settings-promise 60 TimeUnit/SECONDS)
     (let [send-message (atom nil)
           ^ChannelHandler stream-channel-handler (stream-channel-handler
                                                   {:host host
                                                    :port port
                                                    :path path
                                                    :max-frame-size (Math/pow 2 16)
                                                    :send-message send-message})]
       (-> (Http2StreamChannelBootstrap. channel)
           (.handler stream-channel-handler)
           .open
           .syncUninterruptibly
           .getNow)
       {:close (fn [] (.shutdownGracefully worker-group))
        :send (fn [message]
                (if-let [send-fn @send-message]
                  (send-fn message)
                  (throw (ex-info "Send handler is not yet registered" {}))))}))))

(defn unsafe-self-signed-ssl-context
  []
  (-> (SslContextBuilder/forClient)
      (.sslProvider SslProvider/JDK)
      (.ciphers Http2SecurityUtil/CIPHERS
                SupportedCipherSuiteFilter/INSTANCE)
      (.trustManager InsecureTrustManagerFactory/INSTANCE)
      (.applicationProtocolConfig (ApplicationProtocolConfig.
                                   ApplicationProtocolConfig$Protocol/ALPN
                                   ApplicationProtocolConfig$SelectorFailureBehavior/NO_ADVERTISE
                                   ApplicationProtocolConfig$SelectedListenerFailureBehavior/ACCEPT
                                   ^"[Ljava.lang.String;" (into-array [ApplicationProtocolNames/HTTP_2])))
      (.build)))
