;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.netty.http.client.websocket
  (:refer-clojure :exclude [send])
  (:require [via.defaults :refer [default-via-endpoint]])
  (:import [java.util HashMap]
           [java.util.concurrent TimeUnit]
           [io.netty.handler.codec.http2 Http2SecurityUtil]
           [io.netty.bootstrap Bootstrap]
           [io.netty.buffer ByteBuf]
           [io.netty.channel.socket SocketChannel]
           [io.netty.channel
            SimpleChannelInboundHandler
            Channel
            ChannelFuture
            ChannelInitializer
            ChannelDuplexHandler
            ChannelHandler
            ChannelHandlerContext
            ChannelInboundHandlerAdapter
            ChannelOption
            ChannelPromise
            EventLoopGroup]
           [io.netty.handler.codec.http
            FullHttpResponse
            DefaultHttpHeaders
            DefaultFullHttpResponse
            HttpClientCodec
            HttpObjectAggregator
            HttpVersion
            HttpResponseStatus
            HttpRequest
            HttpServerCodec]
           [io.netty.channel.nio NioEventLoopGroup]
           [io.netty.channel.socket.nio NioSocketChannel]
           [io.netty.handler.codec.http.websocketx
            WebSocketHandshakeException
            WebSocketClientHandshakerFactory
            WebSocketClientHandshaker
            WebSocketVersion
            TextWebSocketFrame
            WebSocket13FrameDecoder
            WebSocket13FrameEncoder
            WebSocketDecoderConfig]
           [io.netty.handler.codec.http.websocketx.extensions
            WebSocketClientExtension
            WebSocketExtensionData
            WebSocketExtensionDecoder
            WebSocketExtensionEncoder]
           [io.netty.handler.codec.http.websocketx.extensions.compression
            PerMessageDeflateClientExtensionHandshaker]
           [io.netty.handler.ssl
            SslContextBuilder
            SslContext
            SslProvider
            SupportedCipherSuiteFilter
            ApplicationProtocolConfig
            ApplicationProtocolConfig$Protocol
            ApplicationProtocolConfig$SelectorFailureBehavior
            ApplicationProtocolConfig$SelectedListenerFailureBehavior
            ApplicationProtocolNegotiationHandler
            ApplicationProtocolNames]
           [io.netty.handler.ssl.util
            InsecureTrustManagerFactory]
           [java.net URI]))

(declare websocket-client-handler channel-initializer)

(defn websocket-client
  ([] (websocket-client nil))
  ([{:keys [host port ssl-context path on-text-message]
     :or {host "localhost"
          port 3449
          path default-via-endpoint}}]
   (let [worker-group (NioEventLoopGroup.)]
     (try
       (let [handshake-future (atom nil)
             ^ChannelHandler handler (websocket-client-handler
                                      (WebSocketClientHandshakerFactory/newHandshaker
                                       (URI. (str (if ssl-context
                                                    "wss"
                                                    "ws")
                                                  "://"
                                                  host
                                                  ":"
                                                  port
                                                  (when (not (re-find #"^/" path))
                                                    "/")
                                                  path))
                                       WebSocketVersion/V13
                                       nil
                                       false
                                       (DefaultHttpHeaders.))
                                      handshake-future
                                      on-text-message)
             b (doto (Bootstrap.)
                 (.group worker-group)
                 (.channel ^NioSocketChannel NioSocketChannel)
                 (.handler (channel-initializer host port ssl-context handler)))
             channel (-> b
                         (.connect (str host) (int port))
                         .sync
                         .channel)]
         (.sync ^ChannelPromise @handshake-future)
         {:close (fn [] (.shutdownGracefully worker-group))
          :send (fn [^String message]
                  (.writeAndFlush channel (TextWebSocketFrame. message)))})
       (catch Exception e
         (.shutdownGracefully worker-group)
         (throw e))))))

(defn send
  [{:keys [send] :as ws-client} text-message]
  (send text-message))

(defn close
  [{:keys [close] :as ws-client}]
  (close))

(defn unsafe-self-signed-ssl-context
  []
  (-> (SslContextBuilder/forClient)
      (.sslProvider SslProvider/JDK)
      (.ciphers Http2SecurityUtil/CIPHERS
                SupportedCipherSuiteFilter/INSTANCE)
      (.trustManager InsecureTrustManagerFactory/INSTANCE)
      (.applicationProtocolConfig (ApplicationProtocolConfig.
                                   ApplicationProtocolConfig$Protocol/ALPN
                                   ApplicationProtocolConfig$SelectorFailureBehavior/NO_ADVERTISE
                                   ApplicationProtocolConfig$SelectedListenerFailureBehavior/ACCEPT
                                   ^"[Ljava.lang.String;" (into-array [ApplicationProtocolNames/HTTP_1_1])))
      (.build)))

;;; Private

(defn websocket-client-handler
  ^ChannelDuplexHandler [^WebSocketClientHandshaker handshaker handshake-future on-text-message]
  (proxy [SimpleChannelInboundHandler] []
    (handlerAdded [^ChannelHandlerContext ctx]
      (reset! handshake-future (.newPromise ctx)))
    (channelActive [^ChannelHandlerContext ctx]
      (.handshake handshaker (.channel ctx)))
    (channelInactive [^ChannelHandlerContext ctx]
      ;; client disconnected
      )
    (channelRead0 [^ChannelHandlerContext ctx msg]
      (let [ch (.channel ctx)]
        (cond
          (not (.isHandshakeComplete handshaker))
          (try
            (.finishHandshake handshaker ch ^FullHttpResponse msg)
            ;; client connected
            (.setSuccess ^ChannelPromise @handshake-future)
            (catch WebSocketHandshakeException e
              (.setFailure ^ChannelPromise @handshake-future e)))

          (instance? FullHttpResponse msg)
          (throw (ex-info "Unexpected FullHttpResponse" {:msg msg}))

          :else
          (let [^WebSocketFrame frame msg]
            (cond
              (instance? TextWebSocketFrame msg)
              (let [text (.text ^TextWebSocketFrame msg)]
                (when on-text-message
                  (on-text-message text)))
              :else (println "Unhandled WebSocket frame" {:frame frame}))))))
    (exceptionCaught [^ChannelHandlerContext ctx ^Throwable cause]
      (when (not (.isDone ^ChannelPromise @handshake-future))
        (.setFailure ^ChannelPromise @handshake-future cause))
      (.close ctx))))

(defn channel-initializer
  ^ChannelInitializer [^String host ^long port ^SslContext ssl-context ^ChannelHandler handler]
  (proxy [ChannelInitializer] []
    (initChannel [^SocketChannel ch]
      (let [p (.pipeline ch)]
        (when ssl-context
          (.addLast p "ssl-handler" (.newHandler ssl-context (.alloc ch) host (int port))))
        (.addLast p "client-codec" (HttpClientCodec.))
        (.addLast p "http-object-aggregator" (HttpObjectAggregator. 8192))
        (.addLast p "client-handler" handler)))))
