;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.
(ns via.endpoint
  (:refer-clojure :exclude [send])
  (:require #?(:cljs [via.adapters.haslett :as haslett])
            #?(:cljs [utilis.js :as j])
            [via.adapter :as adapter]
            #?(:clj [via.adapters.netty :as netty])
            [via.util.id :refer [uuid]]
            [via.defaults :as defaults]
            [signum.fx :as sfx]
            [signum.events :as se]
            [signum.subs :as ss]
            [tempus.core :as t]
            [tempus.transit :as tt]
            [cognitect.transit :as transit]
            [utilis.fn :refer [fsafe]]
            [utilis.map :refer [map-vals]]
            [utilis.timer :as timer]
            [integrant.core :as ig]
            [clojure.set :refer [union difference]]
            [clojure.string :as st]
            #?(:clj [com.brunobonacci.mulog :as u]))
  #?(:clj (:import [java.io ByteArrayInputStream ByteArrayOutputStream])))

(defonce endpoints (atom #{}))
(def outgoing-connect-lock #?(:clj (Object.) :cljs (js/Object.)))

(declare encode-message decode-message
         connect disconnect cancel-reconnect-task
         handle-connect handle-disconnect
         handle-message handle-event
         normalize-namespace)

(defn endpoint
  [{:keys [peers
           exports
           transit-handlers
           event-listeners
           request-timeout
           context
           params
           client-connect
           handle-connection]
    :or {request-timeout defaults/request-timeout}}]
  (try (let [{:keys [events subs namespaces]} exports
             params (merge params {:via-language #?(:clj :clj :cljs :cljs)})
             endpoint (#?(:cljs haslett/adapter
                          :clj netty/adapter)
                       {:client-connect client-connect
                        :handle-connection handle-connection
                        :event-listeners (atom (map-vals (fn [handler]
                                                           {(uuid) handler})
                                                         event-listeners))
                        :exports (atom {:subs (set subs)
                                        :events (set (concat events
                                                             [:via.session-context/replace
                                                              :via.session-context/merge]))
                                        :namespaces (set namespaces)})
                        :peers (atom {})
                        :params params
                        :requests (atom {})
                        :context (atom (into {} context))
                        :handle-message (fn [& args] (apply handle-message args))
                        :handle-disconnect (fn [& args] (apply handle-disconnect args))
                        :handle-connect (fn [& args] (apply handle-connect args))
                        :decode (partial decode-message {:handlers (merge (:read tt/handlers) (:read transit-handlers))})
                        :encode (partial encode-message {:handlers (merge (:write tt/handlers) (:write transit-handlers))})
                        :request-timeout request-timeout})]
         (swap! endpoints conj endpoint)
         (doseq [peer-address peers]
           (connect endpoint
                    (if (string? peer-address)
                      peer-address
                      (:address peer-address))
                    (uuid)
                    (when (map? peer-address)
                      (:ssl-context peer-address))))
         endpoint)
       (catch #?(:clj Exception :cljs js/Error) e
         (#?(:clj u/log :cljs js/console.error) ::start :exception e)
         (throw e))))

(defn shutdown
  [endpoint]
  (doseq [[peer-id _] @(adapter/peers (endpoint))]
    (disconnect endpoint peer-id))
  (adapter/shutdown (endpoint))
  (swap! endpoints disj endpoint)
  endpoint)

(defmethod ig/init-key :via/endpoint
  [_ args]
  (endpoint args))

(defmethod ig/halt-key! :via/endpoint
  [_ endpoint]
  (shutdown endpoint))

(defn send
  "Asynchronously sends `message` to the client for `peer-id`"
  [endpoint peer-id message & {:keys [type _timeout headers params
                                      on-success
                                      on-failure
                                      on-timeout timeout]
                               :or {type :event
                                    timeout 30000}}]
  (try
    (when (not peer-id)
      (throw (ex-info "No peer-id provided" {:message message
                                             :peer-id peer-id
                                             :type type
                                             :headers headers
                                             :timeout timeout})))
    (let [message (merge (when type {:type type})
                         (when message {:body message})
                         (when (map? headers)
                           {:headers headers})
                         params)]
      (if (or on-success on-failure on-timeout)
        (let [request-id (uuid)
              message (assoc-in message [:headers :request-id] request-id)]
          (swap! (adapter/requests (endpoint)) assoc request-id
                 {:on-success on-success
                  :on-failure on-failure
                  :on-timeout on-timeout
                  :message message
                  :timer (timer/run-after
                          #(do (swap! (adapter/requests (endpoint)) dissoc request-id)
                               (try ((fsafe on-timeout))
                                    (catch #?(:clj Exception :cljs js/Error) e
                                      (#?(:clj u/log :cljs js/console.error)
                                       ::timeout-handler :exception e
                                       ::peer-id peer-id ::message message))))
                          timeout)
                  :timeout timeout
                  :timestamp (t/now)
                  :peer-id peer-id})
          (adapter/send (endpoint) peer-id ((adapter/encode (endpoint)) message)))
        (adapter/send (endpoint) peer-id ((adapter/encode (endpoint)) message))))
    (catch #?(:clj Exception :cljs js/Error) e
      (#?(:clj u/log :cljs js/console.error)
       ::send :exception e
       ::peer-id peer-id
       ::message message))))

(defn send-to-tag
  [endpoint tag message & {:keys [type _timeout headers
                                  on-success
                                  on-failure
                                  on-timeout timeout]
                           :or {type :event
                                timeout 30000}}]
  (doseq [peer-id (->> @(adapter/peers (endpoint))
                       (filter (fn [[_peer-id {:keys [tags]}]]
                                 (get tags tag)))
                       (map first))]
    (send endpoint peer-id message
          :type type
          :timeout timeout
          :headers headers
          :on-success on-success
          :on-failure on-failure
          :on-timeout on-timeout
          :timeout timeout)))

(defn broadcast
  "Asynchronously sends `message` to all connected clients"
  [endpoint message]
  (doseq [[peer-id _] @(adapter/peers (endpoint))]
    (send endpoint peer-id message)))

(defn connected?
  [endpoint peer-id]
  (boolean (get-in @(adapter/peers (endpoint)) [peer-id :connection])))

(defn disconnect
  ([endpoint peer-id]
   (disconnect endpoint peer-id false))
  ([endpoint peer-id reconnect]
   (when (connected? endpoint peer-id)
     (swap! (adapter/peers (endpoint)) update peer-id
            merge {:reconnect reconnect
                   :status (if reconnect
                             :reconnecting
                             :disconnecting)})
     (adapter/disconnect (endpoint) peer-id)
     (when (not reconnect)
       (cancel-reconnect-task endpoint peer-id)))))

(defn first-peer
  [endpoint]
  (->> @(adapter/peers (endpoint))
       (filter (comp #{:connecting
                       :connected
                       :reconnecting}
                     :status
                     second))
       ffirst))

(defn connected-peers
  [endpoint]
  (->> @(adapter/peers (endpoint))
       vals
       (filter (comp (partial = :connected) :status))))

#?(:cljs (defn peer-connected-states
           [endpoint]
           (let [result (volatile! (transient []))]
             (doseq [{:keys [id connection] :as peer} (vals @(adapter/peers (endpoint)))]
               (let [{:keys [socket]} connection
                     state (case (j/get socket :readyState)
                             0 :connecting
                             1 :connected
                             2 :closing
                             3 :disconnected)]
                 (when (not= state (:status peer))
                   (let [peer (get (swap! (adapter/peers (endpoint)) assoc-in [(:id peer) :status] state)
                                   (:id peer))]
                     (condp = state
                       :connected (handle-event endpoint :via.endpoint.peer/connected peer)
                       :disconnected (handle-event endpoint :via.endpoint.peer/disconnected peer)
                       nil)))
                 (vswap! result conj!
                         {:id id
                          :status state})))
             (persistent! @result))))

(defn first-endpoint
  []
  (first @endpoints))

(defn add-event-listener
  [endpoint key listener]
  (adapter/add-event-listener (endpoint) key listener))

(defn remove-event-listener
  [endpoint key listener-id]
  (adapter/remove-event-listener (endpoint) key listener-id))

(defn handle-event
  [endpoint key event]
  (let [handlers (get @(adapter/event-listeners (endpoint)) key)
        default-handlers (get @(adapter/event-listeners (endpoint)) :default)]
    (doseq [[_ handler] (concat handlers default-handlers)]
      (handler [key event]))
    (boolean (seq handlers))))

(defn session-context
  ([] (session-context (first @endpoints)))
  ([endpoint] (session-context endpoint (first-peer endpoint)))
  ([endpoint peer-id]
   (-> @(adapter/peers (endpoint))
       (get peer-id)
       :session-context)))

(defn update-session-context
  ([endpoint peer-id f]
   (update-session-context endpoint peer-id true f))
  ([endpoint peer-id sync f]
   (let [peers (adapter/peers (endpoint))]
     (when (contains? @peers peer-id)
       (let [session-context (-> peers
                                 (swap! update-in [peer-id :session-context] f)
                                 (get peer-id)
                                 :session-context)]
         (handle-event endpoint :via.endpoint.session-context/change session-context)
         (when sync
           (let [event-handler-args {:peer-id peer-id :session-context session-context}]
             (send endpoint peer-id [:via.session-context/replace {:session-context session-context :sync false}]
                   :on-success #(handle-event endpoint :via.endpoint.session-context.update/on-success (assoc event-handler-args :reply %))
                   :on-failure #(handle-event endpoint :via.endpoint.session-context.update/on-failure (assoc event-handler-args :reply %))
                   :on-timeout #(handle-event endpoint :via.endpoint.session-context.update/on-timeout event-handler-args)
                   :timeout (adapter/opt (endpoint) :request-timeout)))))))))

(defn merge-context
  [endpoint context]
  (swap! (adapter/context (endpoint)) merge context))

(defn export-sub
  [endpoint sub-id]
  (swap! (adapter/exports (endpoint)) update :subs conj sub-id))

(defn unexport-sub
  [endpoint sub-id]
  (swap! (adapter/exports (endpoint)) update :subs disj sub-id))

(defn sub?
  [endpoint sub-id]
  (boolean
   (or (get-in @(adapter/exports (endpoint)) [:subs sub-id])
       (some #(= (normalize-namespace %)
                 (normalize-namespace (ss/namespace sub-id)))
             (:namespaces @(adapter/exports (endpoint)))))))

(defn export-event
  [endpoint event-id]
  (swap! (adapter/exports (endpoint)) update :events conj event-id))

(defn unexport-event
  [endpoint event-id]
  (swap! (adapter/exports (endpoint)) update :events disj event-id))

(defn event?
  [endpoint event-id]
  (boolean
   (or (get-in @(adapter/exports (endpoint)) [:events event-id])
       (some #(= (normalize-namespace %)
                 (normalize-namespace (se/namespace event-id)))
             (:namespaces @(adapter/exports (endpoint)))))))

(defn connect
  ([endpoint peer-address]
   (connect endpoint peer-address (uuid)))
  ([endpoint peer-address peer-id]
   (connect endpoint peer-address peer-id nil))
  ([endpoint peer-address peer-id ssl-context]
   (locking outgoing-connect-lock
     (let [peer {:id peer-id
                 :role :originator
                 :request {:peer-id peer-id
                           :peer-address peer-address}
                 :status :connecting}]
       (swap! (adapter/peers (endpoint)) assoc (:id peer) peer)
       #?(:clj (if-let [connection (adapter/connect (endpoint) peer-address ssl-context)]
                 (do ((adapter/handle-connect (endpoint)) endpoint (merge peer {:connection connection
                                                                                :status :connected}))
                     peer-id)
                 (do (swap! (adapter/peers (endpoint)) dissoc (:id peer))
                     nil))
          :cljs (-> (endpoint)
                    (adapter/connect peer-address ssl-context)
                    (j/call :then (fn [connection]
                                    ((adapter/handle-connect (endpoint)) endpoint (merge peer {:connection connection
                                                                                               :status :connected}))
                                    peer-id))
                    (j/call :catch (fn [error]
                                     (swap! (adapter/peers (endpoint)) dissoc (:id peer))
                                     (throw error)))))))))

(defn send-reply
  [endpoint peer-id request-id {:keys [status body]}]
  (send endpoint peer-id body
        :type :reply
        :params {:status status}
        :headers {:request-id request-id}))

;;; Effect Handlers

(sfx/reg-fx
 :via/disconnect
 (fn [{:keys [endpoint request]} {:keys [peer-ids] :as opts}]
   (doseq [peer-id (or (not-empty peer-ids)
                       [(:peer-id request)])]
     (when-let [message (:via/send opts)]
       (send endpoint peer-id message))
     (disconnect endpoint peer-id))))

(sfx/reg-fx
 :via.tags/add
 (fn [{:keys [endpoint request]} {:keys [tags]}]
   (swap! (adapter/peers (endpoint)) update-in [(:peer-id request) :tags] #(union (set %) (set tags)))))

(sfx/reg-fx
 :via.tags/remove
 (fn [{:keys [endpoint request]} {:keys [tags]}]
   (swap! (adapter/peers (endpoint)) update-in [(:peer-id request) :tags] #(difference (set %) (set tags)))))

(sfx/reg-fx
 :via.tags/replace
 (fn [{:keys [endpoint request]} {:keys [tags]}]
   (swap! (adapter/peers (endpoint)) assoc-in [(:peer-id request) :tags] (set tags))))

(sfx/reg-fx
 :via/reply
 (fn [{:keys [endpoint request event message]} {:keys [status body] :as reply}]
   (when (not status)
     (throw (ex-info "A status must be provided in a :via/reply"
                     {:eg {:status 200}})))
   (if-let [request-id (-> message :headers :request-id)]
     (send-reply endpoint (:peer-id request) request-id reply)
     (handle-event endpoint :via.endpoint.outbound-reply/unhandled
                   {:reply (merge {:type :reply
                                   :reply-to event
                                   :status status}
                                  (when body {:body body}))}))))

(sfx/reg-fx
 :via/send
 (fn [{:keys [endpoint]} {:keys [peer-id message headers]}]
   (send endpoint peer-id message
         :headers headers)))

(sfx/reg-fx
 :via.session-context/replace
 (fn [{:keys [endpoint request]} {:keys [session-context sync]}]
   (update-session-context endpoint (:peer-id request) sync (constantly session-context))))

(sfx/reg-fx
 :via.session-context/merge
 (fn [{:keys [endpoint request]} {:keys [session-context sync]}]
   (update-session-context endpoint (:peer-id request) sync #(merge % session-context))))

;;; Event Handlers

(se/reg-event
 :via.session-context/replace
 (fn [_ [_ {:keys [session-context sync]}]]
   {:via.session-context/replace {:session-context session-context
                                  :sync sync}
    :via/reply {:status 200}}))

(se/reg-event
 :via.session-context/merge
 (fn [_ [_ {:keys [session-context sync]}]]
   {:via.session-context/merge {:session-context session-context
                                :sync sync}
    :via/reply {:status 200}}))

;;; Implementation

(defn- encode-message
  [handlers data]
  #?(:clj (let [out (ByteArrayOutputStream. 4096)]
            (transit/write (transit/writer out :json handlers) data)
            (.toString out))
     :cljs (transit/write (transit/writer :json handlers) data)))

(defn- decode-message
  [handlers ^String data]
  #?(:clj (let [in (ByteArrayInputStream. (.getBytes data))]
            (transit/read (transit/reader in :json handlers)))
     :cljs (transit/read (transit/reader :json handlers) data)))

(defn- handle-reply
  [endpoint reply]
  (let [request-id (-> reply :headers :request-id)]
    (if-let [request (and request-id (get @(adapter/requests (endpoint)) request-id))]
      (do ((fsafe timer/cancel) (:timer request))
          (when-let [f (get {200 (:on-success request)} (:status reply) (:on-failure request))]
            (try (f reply)
                 (catch #?(:clj Exception :cljs js/Error) e
                   (#?(:clj u/log :cljs js/console.error) ::reply :exception e ::reply reply))))
          (swap! (adapter/requests (endpoint)) dissoc request-id))
      (handle-event endpoint :via.endpoint.inbound-reply/unhandled {:reply reply}))))

(defn- send-unknown-event-reply
  [endpoint peer-id message]
  (when-let [request-id (-> message :headers :request-id)]
    (send-reply endpoint peer-id
                request-id {:status 400
                            :body {:error :via.endpoint/unknown-event
                                   :event (:body message)}})))

(defn- handle-remote-event
  [endpoint request {:keys [body] :as message}]
  (let [[event-id & _ :as event] body
        exported? (event? endpoint event-id)
        se-event? (se/event? event-id)
        known-event? se-event?]
    (if (or (not exported?) (not known-event?))
      (do (handle-event endpoint :via.endpoint/unknown-event {:message message})
          (send-unknown-event-reply endpoint (:peer-id request) message))
      (se/dispatch {:endpoint endpoint
                    :request request
                    :message message
                    :event event
                    :peer (get @(adapter/peers (endpoint)) (:peer-id request))} event))))

(defn- handle-message
  [endpoint request message]
  (try (when-let [message ((adapter/decode (endpoint)) message)]
         (condp = (:type message)
           :event (handle-remote-event endpoint request message)
           :reply (handle-reply endpoint message)
           (handle-event endpoint :via.endpoint/unhandled-message {:message message})))
       (catch #?(:clj Exception :cljs js/Error) e
         (when (not (handle-event endpoint :via.endpoint/unhandled-exception e))
           (#?(:clj u/log :cljs js/console.log) ::handle-message :exception e ::message message ::request request)))))

(defn- handle-connect
  [endpoint peer]
  (swap! (adapter/peers (endpoint)) assoc (:id peer)
         (assoc peer
                :connected-timestamp (t/now)
                :status :connected))
  (handle-event endpoint :via.endpoint.peer/connected peer))

(defn- cancel-reconnect-task
  [endpoint peer-id]
  (when-let [reconnect-task (get-in @(adapter/peers (endpoint)) [peer-id :reconnect-task])]
    (timer/cancel reconnect-task))
  (swap! (adapter/peers (endpoint)) update peer-id dissoc :reconnect-task))

(defn- remove-peer
  [endpoint peer-id]
  (let [peer (get @(adapter/peers (endpoint)) peer-id)]
    (cancel-reconnect-task endpoint peer-id)
    (swap! (adapter/peers (endpoint)) dissoc peer-id)
    (handle-event endpoint :via.endpoint.peer/removed peer)))

(defn- reconnect
  ([endpoint peer-address peer-id]
   (reconnect endpoint peer-address peer-id 50))
  ([endpoint peer-address peer-id interval]
   (swap! (adapter/peers (endpoint)) assoc-in [peer-id :status] :reconnecting)
   (let [on-connect-failed (fn []
                             (swap! (adapter/peers (endpoint))
                                    assoc-in [peer-id :reconnect-task]
                                    (timer/run-after
                                     #(reconnect endpoint peer-address peer-id
                                                 (min (* 2 interval)
                                                      defaults/max-reconnect-interval))
                                     (min interval defaults/max-reconnect-interval))))]
     (cancel-reconnect-task endpoint peer-id)
     #?(:clj (if (not (connect endpoint peer-address peer-id))
               (on-connect-failed)
               (cancel-reconnect-task endpoint peer-id))
        :cljs (-> endpoint
                  (connect peer-address peer-id)
                  (j/call :then (fn [_] (cancel-reconnect-task endpoint peer-id)))
                  (j/call :catch (fn [_] (on-connect-failed))))))))

(defn- handle-disconnect
  [endpoint {:keys [role request] :as peer}]
  (handle-event endpoint :via.endpoint.peer/disconnected peer)
  (if (and (= :originator role)
           (not (false? (:reconnect peer))))
    (reconnect endpoint (:peer-address request) (:id peer))
    (remove-peer endpoint (:id peer))))

(def namespace-type (type *ns*))

(defn namespace?
  [ns]
  (instance? namespace-type ns))

(defn- normalize-namespace
  [ns]
  (cond
    (nil? ns) nil
    (string? ns) (keyword
                  (-> (str ns)
                      (st/replace #"^:" "")
                      (st/replace #"/" ".")))
    (keyword? ns) (normalize-namespace (str ns))
    (namespace? ns) #?(:clj (keyword (.getName ^clojure.lang.Namespace ns))
                       :cljs (keyword (j/call ns :getName)))
    :else (throw (ex-info "Can't normalize namespace" {:ns ns}))))
