;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.netty.http.server.server
  (:require [via.util.netty :as n]
            [clj-commons.byte-streams :as bs]
            [clojure.string :as st]
            [com.brunobonacci.mulog :as u]
            [utilis.map :refer [map-keys]])
  (:import [io.netty.channel
            ChannelDuplexHandler
            ChannelPipeline
            ChannelHandler
            ChannelHandlerContext]
           [java.util.concurrent ExecutorService]
           [io.netty.util ReferenceCounted]
           [io.netty.handler.codec.http
            DefaultHttpHeaders
            DefaultFullHttpResponse
            HttpVersion
            HttpResponseStatus
            HttpRequest
            HttpServerCodec]
           [io.netty.handler.ssl SslHandler]
           [io.netty.handler.codec.http.websocketx
            WebSocketServerHandshakerFactory
            WebSocketFrame
            TextWebSocketFrame
            BinaryWebSocketFrame
            CloseWebSocketFrame]))

(defn websocket-request?
  [req]
  (boolean
   (and (= "websocket" (st/lower-case (str (get-in req [:headers "upgrade"]))))
        (= "upgrade" (st/lower-case (str (get-in req [:headers "connection"])))))))

(defn http1-request
  [^ChannelHandlerContext ctx ^HttpRequest msg]
  (let [headers (->> (.headers msg)
                     .entries
                     (map (fn [[k v]]
                            [(st/lower-case (str k)) (str v)]))
                     (into {}))
        path (.uri msg)
        [uri query-string] (st/split path #"\?")
        peer-cert-chain (try
                          (when-let [^SslHandler ssl-handler (some-> ctx .channel .pipeline (.get "ssl-handler"))]
                            (some-> ssl-handler .engine .getSession
                                    .getPeerCertificates seq))
                          (catch Exception _ nil))]
    {:headers headers
     :peer-cert-chain peer-cert-chain
     :uri uri
     :path path
     :protocol-version (-> (.protocolVersion msg)
                           .text
                           st/lower-case)
     :scheme (if (websocket-request? {:headers headers})
               :wss
               :https)
     :query-string query-string
     :request-method (-> (.method msg)
                         .name
                         st/lower-case
                         keyword)
     :query-params (when query-string
                     (->> (st/split query-string #"&")
                          (map #(st/split % #"\="))
                          (into {})))}))

(defn test-array
  [t]
  (let [check (type (t []))]
    (fn [arg] (instance? check arg))))

(def byte-array?
  (test-array byte-array))

(defn websocket-message-coerce-fn
  [msg]
  (condp instance? msg
    WebSocketFrame
    msg

    CharSequence
    (TextWebSocketFrame. (bs/to-string msg))

    (BinaryWebSocketFrame. (n/to-byte-buf msg))))

(defn configure-websocket-handlers
  [^ChannelHandlerContext ctx ^ExecutorService exec-service
   {:keys [on-open on-close
           on-text-message
           on-binary-message]}]
  (doto (.pipeline ctx)
    (.remove "request-handler")
    (.addLast "frame-handler"
              (proxy [ChannelDuplexHandler] []
                (channelRead [^ChannelHandlerContext ctx msg]
                  (let [handler (cond
                                  (instance? TextWebSocketFrame msg)
                                  (let [text (.text ^TextWebSocketFrame msg)]
                                    #(on-text-message text))

                                  (instance? BinaryWebSocketFrame msg)
                                  (let [content (.content ^BinaryWebSocketFrame msg)
                                        bytes (byte-array (.readableBytes content))]
                                    (.readBytes content bytes)
                                    #(on-binary-message bytes))

                                  (instance? CloseWebSocketFrame msg)
                                  on-close

                                  :else
                                  (locking Object
                                    (println (ex-info "Unhandled message" {:msg msg}))))]
                    (try (when handler
                           (.submit exec-service
                                    (reify Runnable
                                      (run [_]
                                        (try
                                          (handler)
                                          (catch Exception e
                                            (u/log ::websocket-handler-config :exception e)))))))
                         (finally
                           (when (instance? ReferenceCounted msg)
                             (.release ^ReferenceCounted msg)))))))))
  (.submit exec-service
           (reify Runnable
             (run [_]
               (try
                 (on-open {:send #(n/safe-execute
                                   ctx (fn []
                                         (if-let [frame (websocket-message-coerce-fn %)]
                                           (.writeAndFlush (.channel ctx) frame)
                                           (do (locking Object
                                                 (println (ex-info "Unsupported websocket message type"
                                                                   {:message %})))
                                               (throw (ex-info "Unsupported websocket message type"
                                                               {:message %}))))))
                           :close (fn [] (n/safe-execute ctx (fn [] (.close (.channel ctx)))))})
                 (catch Exception e
                   (u/log ::websocket-handler-config :exception e)))))))

(defn handle-websocket-handshake
  [^ChannelHandlerContext ctx ^HttpRequest req]
  (let [websocket-url (str "wss://"
                           (.get (.headers req) "Host")
                           (.uri req))
        ws-factory (WebSocketServerHandshakerFactory. websocket-url nil true)
        handshaker (.newHandshaker ws-factory req)]
    (if (not handshaker)
      (WebSocketServerHandshakerFactory/sendUnsupportedVersionResponse (.channel ctx))
      (.handshake handshaker (.channel ctx) req))))

(defn http1-request-handler
  ^ChannelHandler [^ExecutorService exec-service handler]
  (proxy [ChannelDuplexHandler] []
    (channelRead [^ChannelHandlerContext ctx msg]
      (if (instance? HttpRequest msg)
        (let [^HttpRequest msg msg
              req (http1-request ctx msg)]
          (or (when (websocket-request? req)
                (let [response (handler req)]
                  (when (= 101 (:status response))
                    (let [{:keys [handlers]} (meta response)]
                      (handle-websocket-handshake ctx msg)
                      (configure-websocket-handlers ctx exec-service handlers))
                    true)))
              (n/run exec-service
                (fn []
                  (try (let [response (handler req)
                             netty-response (DefaultFullHttpResponse.
                                             HttpVersion/HTTP_1_1
                                             (HttpResponseStatus/valueOf (:status response))
                                             (n/to-byte-buf (:body response))
                                             (let [headers (DefaultHttpHeaders.)]
                                               (doseq [[^String k v] (map-keys #(st/lower-case
                                                                                 (str (if (keyword? %)
                                                                                        (name %)
                                                                                        %)))
                                                                               (:headers response))]
                                                 (.add headers k (str v)))
                                               headers)
                                             (DefaultHttpHeaders.))]
                         (n/safe-execute ctx #(.writeAndFlush ctx netty-response)))
                       (catch Exception e
                         (locking Object
                           (println e))))))))
        (let [^ChannelDuplexHandler this this]
          (proxy-super channelRead ctx msg))))))

(defn configure-http1-request-handler
  [^ChannelPipeline pipeline ^ExecutorService exec-service handler]
  (doto pipeline
    (.addLast "request-handler" (http1-request-handler exec-service handler))))

(defn configure-http1-pipeline
  [^ChannelPipeline pipeline ^ExecutorService exec-service handler]
  (doto pipeline
    (.addLast "frame-codec" (HttpServerCodec.))
    (configure-http1-request-handler exec-service handler)))
