;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.adapters.haslett
  (:refer-clojure :exclude [subs])
  (:require [via.adapter :as adapter]
            [via.util.request :refer [encode-query-param]]
            [cljs.core.async :as async]
            [haslett.client :as ws]
            [haslett.format :as fmt]
            [utilis.js :as j]))

(declare connect* disconnect* send* handle-connection)

(defn adapter
  [adapter-opts]
  (let [endpoint (reify adapter/Endpoint
                   (opts [_endpoint] adapter-opts)
                   (send [endpoint peer-id message]
                     (send* endpoint peer-id message))
                   (disconnect [endpoint peer-id]
                     (disconnect* endpoint peer-id))
                   (connect [endpoint address ssl-context]
                     (connect* endpoint address ssl-context))
                   (shutdown [_endpoint]
                     ))]
    (adapter/add-event-listener endpoint :via.endpoint.peer/connected (partial handle-connection endpoint))
    (swap! (adapter/context endpoint) merge
           {::format (reify fmt/Format
                       (read  [_ s] s)
                       (write [_ v] v))})
    (fn ([] endpoint)
      ([request] (throw (ex-info "Unable to handle incoming requests" {:request request}))))))

;;; Implementation

(defn- send*
  [endpoint peer-id message]
  (if message
    (if-let [sink (get-in @(adapter/peers endpoint) [peer-id :connection :sink])]
      (async/put! sink message)
      (js/console.warn "No connection for peer"
                       #js {:peer-id peer-id
                            :message message}))
    (js/console.warn "Tried to put nil message on channel" #js {:peer-id peer-id})))

(defn- append-query-params
  [address params]
  (if (seq params)
    (let [url (js/URL. address)
          origin (j/get url :origin)
          pathname (j/get url :pathname)
          search-params (j/get url :searchParams)]
      (doseq [[key value] params]
        (j/call search-params :append (name key) (encode-query-param value)))
      (str origin pathname "?" search-params))
    address))

(defn- connect*
  [endpoint address _ssl-context]
  (js/Promise.
   (fn [resolve reject]
     (async/go
       (try
         (let [return (-> address
                          (append-query-params (:params (adapter/opts endpoint)))
                          (ws/connect {:format (::format @(adapter/context endpoint))}))
               stream (async/<! return)]
           (if (ws/connected? stream)
             (resolve stream)
             (reject)))
         (catch js/Error e
           (js/console.error "Error occurred in via.adapters.haslett/connect*" e)))))))

(defn- disconnect*
  [endpoint peer-id]
  (when-let [connection (get-in @(adapter/peers endpoint) [peer-id :connection])]
    (ws/close connection)))

(defn- handle-connection
  [endpoint [_ {:keys [connection request]}]]
  (let [{:keys [peer-id]} request
        {:keys [close-status source]} connection]
    (when (not peer-id)
      (throw (ex-info "No peer-id on request"
                      {:request request})))
    (async/go
      (try (loop []
             (async/alt!
               source ([message]
                       (do ((adapter/handle-message endpoint) (constantly endpoint) (assoc request :peer-id peer-id) message)
                           (recur)))
               close-status ([_status]
                             ((adapter/handle-disconnect endpoint) (constantly endpoint) (get @(adapter/peers endpoint) peer-id)))))
           (catch js/Error e
             (js/console.error "Error occurred in via.adapters.haslett/handle-connection" e))))))
