;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.netty.http2.handlers.headers
  (:require [via.netty.http2.handlers.websocket :as ws]
            [via.netty.http2.handlers.data :as data]
            [via.util.netty :as n]
            [utilis.map :refer [map-keys]]
            [utilis.types.number :refer [string->long]]
            [clojure.string :as st])
  (:import [java.util Map$Entry]
           [java.util.concurrent ExecutorService]
           [java.io PipedOutputStream PipedInputStream File]
           [io.netty.buffer ByteBuf]
           [io.netty.channel
            ChannelHandlerContext
            ChannelPipeline
            ChannelPromise]
           [java.net URL]
           [io.netty.handler.stream ChunkedFile]
           [io.netty.handler.codec.http2
            Http2Error
            Http2HeadersFrame
            Http2Headers
            Http2FrameStream
            DefaultHttp2DataFrame
            DefaultHttp2Headers
            DefaultHttp2HeadersFrame
            DefaultHttp2ResetFrame
            Http2DataFrame
            Http2CodecUtil]))

(defn add-all-headers
  [^Http2Headers headers headers-map]
  (doseq [[k v] (map-keys #(st/lower-case
                            (str (if (keyword? %)
                                   (name %)
                                   %)))
                          headers-map)]
    (.add headers k (str v)))
  headers)

(defn headers-frame->headers
  [^Http2HeadersFrame msg]
  (->> (.headers msg)
       .iterator
       iterator-seq
       (reduce (fn [m ^Map$Entry entry]
                 (assoc! m
                         (.toString (.getKey entry))
                         (.toString (.getValue entry))))
               (transient {}))
       persistent!))

(defn request-method
  [headers]
  (case (get headers ":method")
    "GET" :get
    "HEAD" :head
    "POST" :post
    "PUT" :put
    "DELETE" :delete
    "CONNECT" :connect
    "OPTIONS" :options
    "TRACE" :trace
    "PATCH" :patch))

(defn scheme
  [headers]
  (case (get headers ":scheme")
    "http" :http
    "https" :https
    "ws" :ws
    "wss" :wss))

(defn parse-query-string
  [^String query-string]
  (->> (st/split query-string #"&")
       (remove empty?)
       (reduce (fn [query-params kv-string]
                 (let [[k v] (st/split kv-string #"\=")]
                   (assoc! query-params k v)))
               (transient {}))
       persistent!
       not-empty))

(defn uri
  [headers]
  (-> headers
      (get ":path")
      (st/split #"\?")
      first))

(defn create-body-input-stream
  [request]
  (let [content-length (get-in [request :headers] "content-length")
        content-length (when (string? content-length)
                         (string->long content-length))
        in (PipedInputStream. (int (or content-length Http2CodecUtil/DEFAULT_MAX_FRAME_SIZE)))]
    (assoc request
           :body in
           :body-output-stream (PipedOutputStream. in))))

(defn headers-frame->request
  [^Http2HeadersFrame msg]
  (let [headers (headers-frame->headers msg)
        ^String path (get headers ":path")
        url (URL. ^String (get headers ":scheme")
                  ^String (get headers ":authority")
                  path)
        request-method (request-method headers)
        query-string (.getQuery url)]
    (cond-> {:headers headers
             :protocol-version "http/2"
             :uri (uri headers)
             :path path
             :scheme (scheme headers)
             :query-string query-string
             :request-method request-method
             :query-params (when query-string (parse-query-string query-string))}
      (#{:put :post} request-method) (create-body-input-stream))))

(defn handle?
  [msg]
  (instance? Http2HeadersFrame msg))

(defn response->headers
  [{:keys [status headers]}]
  (doto (DefaultHttp2Headers.)
    (.status (str status))
    (add-all-headers headers)))

(defn response->response-headers-frame
  ^Http2HeadersFrame [_connection-state ^Http2FrameStream stream request {:keys [body] :as response}]
  (-> (response->headers response)
      (DefaultHttp2HeadersFrame.
       (boolean
        (and (not (ws/websocket-request? request))
             (nil? body))))
      (.stream stream)))

(defn file->data-frames
  [^ChannelHandlerContext ctx ^Http2FrameStream stream ^File file ^long max-frame-size]
  (let [chunked-file (ChunkedFile. ^File file max-frame-size)
        allocator (.alloc ctx)
        next-frame (fn next-frame []
                     (when (not (.isEndOfInput chunked-file))
                       (lazy-seq
                        (cons (-> (.readChunk chunked-file allocator)
                                  (DefaultHttp2DataFrame.
                                   (boolean (.isEndOfInput chunked-file)))
                                  (.stream stream))
                              (next-frame)))))]
    (next-frame)))

(defn response->response-data-frames
  [connection-state ^ChannelHandlerContext ctx stream _request {:keys [body]}]
  (when (not (nil? body))
    (let [max-frame-size (get-in @connection-state [:settings :settings-max-frame-size])]
      (if (instance? File body)
        (file->data-frames ctx stream ^File body max-frame-size)
        (let [^ByteBuf byte-buf (n/to-byte-buf ctx body)
              frames (->> max-frame-size
                          (n/byte-buf-to-http2-data-frames stream byte-buf)
                          (mapv n/acquire))]
          (n/release byte-buf)
          frames)))))

(defn setup-stream-window
  [connection-state ^ChannelPipeline pipeline ^Http2HeadersFrame msg stream-id request]
  (let [window-size (get-in @connection-state [:settings :settings-initial-window-size])
        backlog (volatile! [])
        cleanup-handlers (atom [(fn []
                                  (->> (-> Http2Error/STREAM_CLOSED
                                           (DefaultHttp2ResetFrame.)
                                           (.stream (.stream msg)))
                                       (n/invoke-write pipeline "frame-writer"))
                                  (doseq [[_ ^Http2DataFrame msg _] @backlog]
                                    (try (n/release msg)
                                         (catch Exception e
                                           (println "Exception releasing message from backlog" e))))
                                  (vreset! backlog nil))])]
    (swap! connection-state update-in
           [:streams stream-id]
           (fn [stream]
             (assoc stream
                    :request request
                    :window (volatile! window-size)
                    :backlog backlog
                    :cleanup-handlers cleanup-handlers
                    :close (fn []
                             (doseq [handler @cleanup-handlers]
                               (try
                                 (handler)
                                 (catch Exception e
                                   (println "Exception occurred in cleanup handler" e))))
                             (swap! connection-state update :streams dissoc stream-id)))))))

(defn setup-websocket-handler
  [^ChannelHandlerContext ctx connection-state ^Http2HeadersFrame msg stream-id response]
  (when-let [data-frame-handler (ws/init-handler ctx connection-state msg response)]
    (swap! connection-state assoc-in
           [:streams stream-id :data-frame-handler]
           data-frame-handler)))

(defn setup-upload-handler
  [^ChannelHandlerContext ctx connection-state ^Http2HeadersFrame msg stream-id request]
  (let [{:keys [body-output-stream]} request
        ^PipedOutputStream body-output-stream body-output-stream
        close-bos (fn []
                    (try (.close body-output-stream)
                         (catch Exception e
                           (println "Exception occurred closing body output stream" e))))
        done-promise (promise)]
    (swap! connection-state assoc-in
           [:streams stream-id :data-frame-handler]
           (fn [^Http2DataFrame frame]
             (let [^ByteBuf buf (.content frame)
                   bytes (byte-array (.readableBytes buf))]
               (.readBytes buf bytes)
               (.write body-output-stream bytes 0 (count bytes))
               (when (.isEndStream frame)
                 (close-bos)
                 (deliver done-promise :done)))))
    (when-let [buffer (get-in @connection-state [:streams stream-id :data-frame-buffer])]
      (n/safe-execute
       ctx
       (fn []
         (doseq [^Http2DataFrame frame @buffer]
           (data/handle ctx connection-state frame)
           (n/release frame))
         (when (get-in @connection-state [:streams stream-id])
           (swap! connection-state update-in [:streams stream-id]
                  dissoc :data-frame-buffer)))))
    done-promise))

(defn handle
  [^ChannelHandlerContext ctx connection-state ^Http2HeadersFrame msg]
  (let [{:keys [handler exec-service]} @connection-state
        ^ExecutorService exec-service exec-service
        ^ChannelPipeline pipeline (.pipeline ctx)
        stream (.stream msg)
        stream-id (.id stream)
        upload? (boolean (#{"PUT" "POST"} (str (.get (.headers msg) ":method"))))]
    (when upload?
      (swap! connection-state assoc-in
             [:streams stream-id :data-frame-buffer]
             (volatile! [])))
    (n/run exec-service
      (fn []
        (try (let [request (headers-frame->request msg)
                   window? (boolean
                            (or (ws/websocket-request? request)
                                (#{:put :post} (:request-method request))))
                   _ (when window? (setup-stream-window connection-state pipeline msg stream-id request))
                   done-upload (when upload? (setup-upload-handler ctx connection-state msg stream-id request))
                   response (handler request)
                   response-headers-frame (response->response-headers-frame connection-state stream request response)
                   response-data-frames (response->response-data-frames connection-state ctx stream request response)
                   response-byte-count (reduce (fn [c ^Http2DataFrame frame]
                                                 (+ c (.initialFlowControlledBytes frame)))
                                               0
                                               response-data-frames)
                   window? (boolean
                            (or window?
                                (> response-byte-count
                                   (get-in @connection-state [:settings :settings-initial-window-size]))))]
               (when (and window? (not (get-in @connection-state [:streams stream-id :window])))
                 (setup-stream-window connection-state pipeline msg stream-id request))
               (when (ws/websocket-request? request)
                 (setup-websocket-handler ctx connection-state msg stream-id response))
               (when done-upload @done-upload)
               (n/safe-execute
                ctx
                (fn []
                  (let [byte-counter (volatile! 0)
                        close-stream (fn [^ChannelPromise p]
                                       (n/with-promise-listener p
                                         (fn []
                                           (when-let [close (get-in @connection-state [:streams stream-id :close])]
                                             (n/safe-execute ctx (fn [] (close)))))))]
                    (let [^Http2HeadersFrame frame response-headers-frame
                          ^ChannelPromise p (n/invoke-write pipeline "frame-writer" frame)]
                      (when (.isEndStream frame) (close-stream p)))
                    (doseq [^Http2DataFrame frame response-data-frames]
                      (let [^ChannelPromise p (n/invoke-write pipeline "frame-writer" frame)]
                        (vswap! byte-counter + (.initialFlowControlledBytes ^Http2DataFrame frame))
                        (when (> @byte-counter 1E5)
                          (vreset! byte-counter 0)
                          (.flush ctx))
                        (when (.isEndStream frame)
                          (close-stream p)))))
                  (.flush ctx))))
             (catch Exception e
               (try (.fireExceptionCaught ctx e)
                    (catch Exception _)))))))
  nil)
