(ns via.netty.http.websocket
  (:import [io.netty.bootstrap Bootstrap]
           [io.netty.channel
            ChannelHandlerContext
            ChannelInitializer
            SimpleChannelInboundHandler]
           [io.netty.handler.codec.http
            DefaultHttpHeaders
            FullHttpResponse
            HttpClientCodec
            HttpObjectAggregator]
           [io.netty.handler.codec.http.websocketx
            WebSocketVersion
            WebSocketClientHandshakerFactory
            WebSocketClientHandshaker
            WebSocketFrame]
           [io.netty.handler.codec.http.websocketx.extensions.compression
            WebSocketClientCompressionHandler]
           [io.netty.channel.nio NioEventLoopGroup]
           [io.netty.channel.socket
            SocketChannel]
           [io.netty.channel.socket.nio
            NioServerSocketChannel]
           [io.netty.handler.ssl
            SslContextBuilder]
           [io.netty.handler.ssl.util
            InsecureTrustManagerFactory]
           [java.net URI]))

(defn websocket-client-handler
  [^WebSocketClientHandshaker handshaker]
  (let [handshake-future (atom nil)]
    (proxy [SimpleChannelInboundHandler] []
      (handlerAdded [^ChannelHandlerContext ctx]
        (prn :foo1)
        (reset! handshake-future (.newPromise ctx)))
      (channelActive [^ChannelHandlerContext ctx]
        (prn :foo2)
        (.handshake handshaker (.channel ctx)))
      (channelInactive [^ChannelHandlerContext ctx]
        (prn :foo3)
        ;; disconnected
        )
      (channelRead [^ChannelHandlerContext ctx msg]
        (prn :foo4.0)
        (let [ch (.channel ctx)]
          (prn :foo4)
          (cond
            (not (.isHandshakeComplete ctx))
            (try (prn :foo5)
                 (.finishHandshake handshaker ch ^FullHttpResponse msg)
                 ;; connected
                 (.setSuccess @handshake-future)
                 (catch Exception e
                   ;; handshake failed
                   (.setFailure @handshake-future)))

            (instance? msg FullHttpResponse)
            (throw (ex-info "Unexpected FullHttpResponse" {:msg msg}))

            :else
            (let [^WebSocketFrame frame msg]
              (prn :frame msg))))))))

(defn connect
  [uri]
  (let [group (NioEventLoopGroup.)
        b (Bootstrap.)
        uri (URI/create uri)
        scheme (or (.getScheme uri) "ws")
        host (or (.getHost uri) "127.0.0.1")
        port (if (= -1 (.getPort uri))
               (condp = scheme
                 "ws" 80
                 "wss" 443
                 -1)
               (.getPort uri))
        _ (when (not (#{"ws" "wss"} scheme))
            (throw (ex-info "Unsupported scheme" {:scheme scheme})))
        ssl? (= "wss" scheme)
        ssl-ctx (when ssl?
                  (-> (SslContextBuilder/forClient)
                      (.trustManager InsecureTrustManagerFactory/INSTANCE)
                      .build))
        handler (websocket-client-handler
                 (WebSocketClientHandshakerFactory/newHandshaker
                  uri
                  WebSocketVersion/V13
                  nil
                  true
                  (DefaultHttpHeaders.)))]
    (-> b
        (.group group)
        (.channel ^Class NioServerSocketChannel)
        (.handler (proxy [ChannelInitializer] []
                    (initChannel [^SocketChannel ch]
                      (let [p (.pipeline ch)]
                        (when ssl-ctx
                          (.addLast p "ssl-handler" (.newHandler ssl-ctx (.alloc ch) host port)))
                        (doto p
                          (.addLast "http-codec" (HttpClientCodec.))
                          (.addLast "aggregator" (HttpObjectAggregator. 8192))
                          (.addLast "compression-handler" WebSocketClientCompressionHandler/INSTANCE)
                          (.addLast "ws-handler" handler)))))))

    (let [ch (-> b
                 (.connect (.getHost uri) port)
                 .sync)]
      (prn :connection ch)
      (fn []
        (.sync (.closeFuture ch))))))

(comment

  (def stop-connection
    (connect "wss://localhost:3449/via"))

  )
