(ns via.example.views
  (:require [ventus.macros :refer [defnc]]
            [ventus.hooks :refer [use-signal]]
            [tempus.core :as t]
            [helix.core :refer [$]]
            [helix.dom :refer [div h2 h3 button input]]
            [helix.hooks :refer [use-state use-effect]]
            [via.core :refer [subscribe dispatch invoke]]
            [utilis.js :as j]
            [clojure.string :as st]
            [cljs.pprint :refer [pprint]]))

(defnc invoke-reply
  []
  (let [last-reply (use-signal (subscribe [:example/invoke-reply]))]
    (div {:style {:padding 24
                  :min-width 300}}
         (h2 "Remote Invoke Example")
         (button {:on-click (fn [_]
                              (invoke [:api.example/echo (update last-reply :counter #(inc (or % 0)))]
                                      {:on-success #(dispatch [:example.invoke-reply/updated (:body %)])
                                       :on-failure #(js/console.error "invoke/failure" %)
                                       :on-timeout #(js/console.error "invoke/timeout" %)}))}
                 "Remote Invoke")
         (div {:style {:margin-top 4}}
              "Last Reply: " (if last-reply (str last-reply) "none")))))

(defnc dispatch-recv
  []
  (let [last-recv (use-signal (subscribe [:example/send-recv]))]
    (div {:style {:padding 24
                  :min-width 300}}
         (h2 "Remote Dispatch Example")
         (button {:on-click (fn [_]
                              (when (or (nil? last-recv)
                                        (and (map? last-recv)
                                             (number? (:counter last-recv))))
                                (dispatch [:api.example/send-recv
                                           (update last-recv :counter #(inc (or % 0)))])))}
                 "Remote Dispatch")
         (div {:style {:margin-top 4}}
              "Last Value: " (if last-recv (str last-recv) "none")))))

(defnc remote-subs
  []
  (div {:style {:padding 24
                :min-width 300}}
       (h2 "Remote Sub Example")
       (div "Auto Incrementing Counter: " (use-signal (subscribe [:api.example/auto-increment-counter])))
       (div "Counter Multiplied by 5: " (use-signal (subscribe [:api.example.auto-increment-counter/multiply 5])))
       (div "A map: " (pr-str (use-signal (subscribe [:api.example/map]))))))

(defnc connection
  []
  (let [status (use-signal (subscribe [:example/connection-status]))]
    (div {:style {:padding 24
                  :min-width 300}}
         (h2 "Connection")
         (div "Status: " (pr-str status)))))

(defnc disconnect-self
  []
  (div {:style {:padding 24
                :min-width 300}}
       (h2 "Disconnect from Remote Peer")
       (button {:on-click (fn [_]
                            (dispatch [:api.example/disconnect-self :disconnect-msg]))}
               "Disconnect")))

(defnc ring-routing
  []
  (div {:style {:padding 24}}
       (h2 "Ring Routing")))

(defnc percentiles
  [{:keys [value tx] :or {tx identity}}]
  (div
   (->> value
        (sort-by first)
        (map (fn [[percentile value]]
               (div {:key (str percentile)}
                    (str percentile ": " (tx value)))))
        (doall))))

(defnc historical
  [{:keys [value tx] :or {tx identity}}]
  (let [{:keys [total]} value]
    (div
     (->> (dissoc value :total)
          (sort-by first)
          (map (fn [[ago value]]
                 (div {:key (str ago)}
                      (str ago "m: " (tx value)))))
          (doall))
     (div "Total: " (str total)))))

(defnc load-test
  []
  (div {:style {:padding 24
                :min-width 300}}
       (h2 "Load Test")
       (button {:on-click (fn [_]
                            (let [received (atom 0)
                                  test-size 1000
                                  f (fn f []
                                      (let [continue (fn []
                                                       (let [received (swap! received inc)]
                                                         (println (str "Completed " received " / " test-size " messages."))
                                                         (if (= test-size received)
                                                           (println "Load test complete.")
                                                           (f))))
                                            msg [@received]]
                                        (-> (invoke [:api.example/echo msg])
                                            (j/call :then #(do (continue)))
                                            (j/call :catch #(do (js/console.error @received %)
                                                                (continue))))))]
                              (f)))}
               "Begin Load Test")))

(defnc send-large-messages
  []
  (let [received-count (use-signal (subscribe [:example.peer/recv-large-message]))]
    (div {:style {:padding 24
                  :min-width 300}}
         (h2 "Send Large Messages over WS")
         (button {:on-click (fn [_] (dispatch [:api.example/send-large-messages]))}
                 "Begin Downloading Large Messages over WS")
         (when (pos? received-count)
           (div (str "Received " received-count " large messages."))))))

(defnc file-upload
  []
  (div {:style {:padding 24
                :min-width 300}}
       (h2 "File Upload")
       (input {:type "file"
               :on-change (fn [event]
                            (let [input (j/get event :currentTarget)
                                  file (j/get-in input [:files 0])]
                              (-> "/upload"
                                  (js/fetch #js {:method "POST"
                                                 :headers #js {"Content-Type" "application/octet-stream"}
                                                 :body file})
                                  (j/call :then #(js/console.log %))
                                  (j/call :catch #(js/console.error %)))))})))

(defnc load-test-test
  []
  ;; start an http1 server from the via example public directory
  ;; npx http-server --cors 127.0.0.1,localhost .
  (let [[file set-file] (use-state nil)]
    (div {:style {:padding 24
                  :min-width 300}}
         (h2 "http/1 vs. h2")
         (input {:placeholder "filename"
                 :on-change #(set-file (j/get-in % [:currentTarget :value]))})
         (button {:on-click (fn [_]
                              (when (seq (st/trim (str file)))
                                (let [test-size 1000
                                      http1-test (fn []
                                                   (js/Promise.
                                                    (fn [resolve _]
                                                      (let [responses (atom [])
                                                            start-time (atom nil)
                                                            complete? (fn [response]
                                                                        (swap! responses conj response)
                                                                        (= test-size (count @responses)))
                                                            complete! (fn []
                                                                        (resolve
                                                                         (- (t/into :long (t/now))
                                                                            (t/into :long @start-time))))
                                                            http1 (fn http1 []
                                                                    (-> (str "http://127.0.0.1:8080/" file)
                                                                        js/fetch
                                                                        (j/call :then #(j/call % :text))
                                                                        (j/call :then (fn [index-html]
                                                                                        (when (complete? index-html)
                                                                                          (complete!))))
                                                                        (j/call :catch #(when (complete? %)
                                                                                          (complete!)))))]
                                                        (reset! start-time (t/now))
                                                        (dotimes [_ test-size]
                                                          (http1))))))
                                      http2-test (fn []
                                                   (js/Promise.
                                                    (fn [resolve _]
                                                      (let [responses (atom [])
                                                            start-time (atom nil)
                                                            complete? (fn [response]
                                                                        (swap! responses conj response)
                                                                        (= test-size (count @responses)))
                                                            complete! (fn []
                                                                        (resolve
                                                                         (- (t/into :long (t/now))
                                                                            (t/into :long @start-time))))
                                                            http2 (fn http2 [i]
                                                                    (-> (str "/" file "?i=" i)
                                                                        (js/fetch #js {:headers #js {"X-i" (str i)}})
                                                                        (j/call :then #(j/call % :text))
                                                                        (j/call :then (fn [index-html]
                                                                                        (when (complete? index-html)
                                                                                          (complete!))))
                                                                        (j/call :catch #(when (complete? %)
                                                                                          (complete!)))))]
                                                        (reset! start-time (t/now))
                                                        (dotimes [i test-size]
                                                          (http2 i))))))]

                                  (-> (http1-test)
                                      (j/call :then (fn [elapsed-ms]
                                                      (js/console.log "http1" test-size "requests in" elapsed-ms "ms")
                                                      (http2-test)))
                                      (j/call :then (fn [elapsed-ms]
                                                      (js/console.log "http2" test-size "requests in" elapsed-ms "ms")
                                                      )))


                                  )))}
                 "Begin Load Test"))))

(defnc file-download
  []
  (div {:style {:padding 24
                :min-width 300}}
       (h2 "Download Large Generated Files")
       (button {:on-click (fn [_]
                            (let [test-size 1000
                                  http2-test (fn []
                                               (js/Promise.
                                                (fn [resolve _]
                                                  (let [responses (atom [])
                                                        start-time (atom nil)
                                                        complete? (fn [response]
                                                                    (swap! responses conj response)
                                                                    (= test-size (count @responses)))
                                                        complete! (fn []
                                                                    (resolve
                                                                     (- (t/into :long (t/now))
                                                                        (t/into :long @start-time))))
                                                        http2 (fn http2 [i]
                                                                (-> (str "/download?i=" i)
                                                                    (js/fetch #js {:headers #js {"X-i" (str i)}})
                                                                    (j/call :then #(j/call % :text))
                                                                    (j/call :then (fn [index-html]
                                                                                    (when (complete? index-html)
                                                                                      (complete!))))
                                                                    (j/call :catch #(when (complete? %)
                                                                                      (complete!)))))]
                                                    (reset! start-time (t/now))
                                                    (dotimes [i test-size]
                                                      (http2 i))))))]

                              (-> (http2-test)
                                  (j/call :then (fn [elapsed-ms]
                                                  (js/console.log "http2" test-size "requests in" elapsed-ms "ms"))))


                              ))}
               "Begin Download Load Test")))

(defnc main-panel
  []
  (div {:style {:padding 24}}
       (div {:style {:display "flex"
                     :flex-direction "row"
                     :justify-content "start"
                     :flex-wrap "wrap"}}
            ($ invoke-reply)
            ($ dispatch-recv)
            ($ remote-subs)
            ($ connection)
            ($ disconnect-self)
            ($ load-test)
            ($ load-test-test)
            ($ send-large-messages)
            ($ file-upload)
            ($ file-download))
       #_($ ring-routing)))
