(ns via.util.netty
  (:require [byte-streams.core :as bs])
  (:import [io.netty.util.concurrent GenericFutureListener]
           [io.netty.channel
            Channel
            ChannelPipeline
            ChannelHandlerContext
            ChannelOutboundHandler
            ChannelInboundHandler
            DefaultChannelPromise
            ChannelPromise]
           [io.netty.util
            ResourceLeakDetector
            ResourceLeakDetector$Level]
           [org.apache.commons.io IOUtils]
           [io.netty.buffer ByteBuf Unpooled]
           [java.nio ByteBuffer]
           [java.io File FileInputStream]
           [java.util.concurrent ExecutorService]
           [io.netty.handler.codec.http2
            Http2Stream
            DefaultHttp2DataFrame]))

(def ^:const array-class (class (clojure.core/byte-array 0)))
(def charset (java.nio.charset.Charset/forName "UTF-8"))

(defn run
  [^ExecutorService exec-service handler & args]
  (.submit exec-service
           (reify Runnable
             (run [_]
               (apply handler args)))))

(defn safe-execute
  [ch f]
  (let [channel (if (instance? Channel ch)
                  ch
                  (.channel ^ChannelHandlerContext ch))
        event-loop (.eventLoop channel)]
    (if (.inEventLoop event-loop)
      (f)
      (.execute event-loop
                (fn []
                  (try
                    (f)
                    (catch Exception e
                      (if (instance? Channel ch)
                        (.fireExceptionCaught (.pipeline ^Channel ch) e)
                        (.fireExceptionCaught ^ChannelHandlerContext ch e))
                      (println "Exception occurred in event-loop" e))))))))

(defn invoke-write
  [^ChannelPipeline pipeline ^String handler-name msg]
  (when-let [writer (.get pipeline handler-name)]
    (let [p (DefaultChannelPromise. (.channel pipeline))]
      (safe-execute
       (.channel pipeline)
       #(.write ^ChannelOutboundHandler writer (.context pipeline handler-name) msg p))
      p)))

(defn invoke-channel-read
  [^ChannelPipeline pipeline ^String handler-name msg]
  (when-let [reader (.get pipeline handler-name)]
    (safe-execute
     (.channel pipeline)
     #(.channelRead ^ChannelInboundHandler reader (.context pipeline handler-name) msg))))

(defn safe-remove-handler
  [^ChannelPipeline pipeline ^String handler-name]
  (safe-execute
   (.channel pipeline)
   #(when ((set (.names pipeline)) handler-name)
      (.remove pipeline handler-name))))

(defn safe-add-handler
  [^ChannelPipeline pipeline handler-name handler]
  (safe-execute
   (.channel pipeline)
   #(if (.get pipeline handler-name)
      (.replace pipeline handler-name handler-name handler)
      (.addLast pipeline handler-name handler))))

(definline release [x]
  `(io.netty.util.ReferenceCountUtil/release ~x))

(definline acquire [x]
  `(io.netty.util.ReferenceCountUtil/retain ~x))

(defn allocate [x]
  (if (instance? Channel x)
    (-> ^Channel x .alloc .ioBuffer)
    (-> ^ChannelHandlerContext x .alloc .ioBuffer)))

(defn append-to-buf! [^ByteBuf buf x]
  (cond
    (instance? array-class x)
    (.writeBytes buf ^bytes x)

    (instance? String x)
    (.writeBytes buf (.getBytes ^String x charset))

    (instance? ByteBuf x)
    (let [b (.writeBytes buf ^ByteBuf x)]
      (release x)
      b)

    :else
    (.writeBytes buf (bs/to-byte-buffer x))))

(defn to-byte-buf
  (^ByteBuf [x]
   (cond
     (nil? x)
     Unpooled/EMPTY_BUFFER

     (instance? array-class x)
     (Unpooled/copiedBuffer ^bytes x)

     (instance? String x)
     (-> ^String x
         (.getBytes charset)
         ByteBuffer/wrap
         Unpooled/wrappedBuffer)

     (instance? ByteBuffer x)
     (Unpooled/wrappedBuffer ^ByteBuffer x)

     (instance? ByteBuf x)
     x

     ;; TODO - this shouldn't read the entire file into memory
     (instance? File x)
     (recur
      (with-open [fis (FileInputStream. ^File x)]
        (IOUtils/toByteArray fis)))

     :else
     (bs/convert x ByteBuf)))
  (^ByteBuf [ch x]
   (if (nil? x)
     Unpooled/EMPTY_BUFFER
     (doto (allocate ch)
       (append-to-buf! x)))))

(defn leak-detector-level! [level]
  (ResourceLeakDetector/setLevel
   (case level
     :disabled ResourceLeakDetector$Level/DISABLED
     :simple ResourceLeakDetector$Level/SIMPLE
     :advanced ResourceLeakDetector$Level/ADVANCED
     :paranoid ResourceLeakDetector$Level/PARANOID)))

(defn slice-byte-buf
  [^ByteBuf byte-buf max-frame-size]
  (let [byte-count (.readableBytes byte-buf)
        slice-count (cond-> (int (/ byte-count max-frame-size))
                      (not (zero? (mod byte-count max-frame-size))) inc)]
    (mapv (fn [slice-index]
            (let [bytes-left (- byte-count (* slice-index max-frame-size))]
              (.slice byte-buf
                      (* slice-index max-frame-size)
                      (min max-frame-size
                           bytes-left))))
          (range slice-count))))

(defn byte-buf-to-http2-data-frames
  ([^Http2Stream stream ^ByteBuf byte-buf max-frame-size]
   (byte-buf-to-http2-data-frames stream byte-buf max-frame-size true))
  ([^Http2Stream stream ^ByteBuf byte-buf max-frame-size terminate?]
   (let [frames (slice-byte-buf byte-buf max-frame-size)]
     (map-indexed
      (fn [index ^ByteBuf frame]
        (-> frame
            (DefaultHttp2DataFrame.
             (boolean
              (and terminate?
                   (= index (dec (count frames))))))
            (.stream stream)))
      frames))))

(defn with-promise-listener
  [p f]
  (.addListener
   p (reify GenericFutureListener
       (operationComplete [_ _]
         (f)))))
