;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.netty.http2.server
  (:require [via.netty.http2.handlers.settings :as settings]
            [via.netty.http2.handlers.unhandled :as unhandled]
            [via.netty.http2.handlers.headers :as headers]
            [via.netty.http2.handlers.data :as data]
            [via.netty.http2.handlers.window :as window]
            [via.netty.http2.handlers.reset :as reset]
            [via.util.netty :as n]
            [tempus.core :as t])
  (:import [java.util.concurrent ExecutorService]
           [io.netty.channel
            ChannelPromise
            ChannelDuplexHandler
            ChannelPipeline
            ChannelHandlerContext]
           [io.netty.util ReferenceCounted]
           [io.netty.handler.codec.http2
            Http2FrameCodecBuilder
            Http2Settings
            Http2DataFrame
            Http2PingFrame
            Http2ResetFrame]))

;; advertise whether websocket is available
(def SETTINGS_ENABLE_CONNECT_PROTOCOL (char 8))

(defn configure-http2-frame-codec-builder
  [^ChannelPipeline pipeline {:keys [initial-window-size
                                     max-frame-size
                                     max-concurrent-streams
                                     max-header-list-size
                                     push-enabled]}]
  (let [settings (Http2Settings/defaultSettings)]
    (.put settings SETTINGS_ENABLE_CONNECT_PROTOCOL (long 1))
    (when initial-window-size (.initialWindowSize initial-window-size))
    (when max-frame-size (.maxFrameSize max-frame-size))
    (when max-concurrent-streams (.maxConcurrentStreams max-concurrent-streams))
    (when max-header-list-size (.maxHeaderListSize max-header-list-size))
    (when push-enabled (.pushEnabled push-enabled))
    (n/safe-add-handler
     pipeline "frame-codec"
     (-> (Http2FrameCodecBuilder/forServer)
         (.autoAckPingFrame true)
         (.initialSettings settings)
         .build))))

(defn configure-http2-inbound-frame-handler
  [^ChannelPipeline pipeline connection-state]
  (n/safe-add-handler
   pipeline "inbound-frame-handler"
   (proxy [ChannelDuplexHandler] []
     (channelRead [^ChannelHandlerContext ctx msg]

       ;; Notes from https://netty.io/wiki/new-and-noteworthy-in-4.0.html#bytebuf-is-always-reference-counted
       ;; When a ByteBuf is used in a ChannelPipeline, there are additional rules you need to keep in mind:
       ;; Each inbound (a.k.a. upstream) handler in a pipeline has to release the received messages. Netty does not release them automatically for you.
       ;; Note that codec framework does release the messages automatically and a user has to increase the reference count if he or she wants to pass a message as-is to the next handler.
       ;; When an outbound (a.k.a. downstream) message reaches at the beginning of the pipeline, Netty will release it after writing it out.

       (try (cond
              (settings/handle? msg)
              (settings/handle ctx connection-state msg)

              (headers/handle? msg)
              (headers/handle ctx connection-state msg)

              (data/handle? msg)
              (data/handle ctx connection-state msg)

              (window/handle? msg)
              (window/handle ctx connection-state msg)

              (reset/handle? msg)
              (reset/handle ctx connection-state msg)

              Http2PingFrame
              (do ) ;; pings are auto-acked

              :else (unhandled/handle ctx connection-state msg))
            (catch Exception e
              (proxy-super exceptionCaught ctx e)))

       (when (instance? ReferenceCounted msg)
         (n/release ^ReferenceCounted msg))))))

(defn configure-exception-handler
  [^ChannelPipeline pipeline _connection-state]
  (n/safe-add-handler
   pipeline "exception-handler"
   (proxy [ChannelDuplexHandler] []
     (exceptionCaught [^ChannelHandlerContext ctx ^Throwable cause]
       (println "Exception occurred in netty pipeline" cause)))))

(defn configure-http2-frame-writer
  [^ChannelPipeline pipeline connection-state]
  (n/safe-add-handler
   pipeline "frame-writer"
   (proxy [ChannelDuplexHandler] []
     (write [^ChannelHandlerContext ctx msg ^ChannelPromise p]
       (cond
         (and (vector? msg) (= :prune (first msg)))
         (let [[_ stream-id] msg
               window (if (zero? stream-id)
                        (:window @connection-state)
                        (get-in @connection-state [:streams stream-id :window]))]
           (when (not window) (throw (ex-info "No window to prune stream" {:msg msg})))
           (when (not (zero? stream-id))
             (let [backlog (get-in @connection-state [:streams stream-id :backlog])
                   flush? (volatile! false)]
               (when (not backlog) (throw (ex-info "No backlog for stream" {:msg msg})))
               (loop []
                 (when (seq @backlog)
                   (let [[^ChannelHandlerContext ctx ^Http2DataFrame msg ^ChannelPromise p] (first @backlog)
                         needed (.initialFlowControlledBytes msg)
                         available @window]
                     (when (>= available needed)
                       (.write ctx msg p)
                       (vreset! flush? true)
                       (vswap! window - needed)
                       (vswap! backlog (comp vec rest))
                       (recur)))))
               (when @flush? (.flush ctx)))))

         (instance? Http2DataFrame msg)
         (let [^Http2DataFrame msg msg
               stream (.stream msg)
               stream-id (.id stream)]
           (if-let [window (get-in @connection-state [:streams stream-id :window])]
             (let [backlog (get-in @connection-state [:streams stream-id :backlog])]
               (when (not backlog) (throw (ex-info "No backlog for stream" {:msg msg})))
               (let [needed (.initialFlowControlledBytes msg)
                     available @window]
                 (if (and (>= available needed)
                          (not (seq @backlog)))
                   (do (.write ctx msg p)
                       (vswap! window - needed))
                   (do (vswap! backlog conj [ctx msg p])
                       (n/invoke-write pipeline "frame-writer" [:prune stream-id])))))
             (.write ctx msg p)))

         :else (.write ctx msg p))))))

(defn configure-http2-pipeline
  [^ChannelPipeline pipeline ^ExecutorService exec-service settings handler]
  (let [connection-state (atom {:exec-service exec-service
                                :handler handler})]
    (doto pipeline
      (configure-http2-frame-codec-builder settings)
      (configure-http2-frame-writer connection-state)
      (configure-http2-inbound-frame-handler connection-state)
      (configure-exception-handler connection-state))))
