(ns via.netty.http2.client
  (:require [utilis.map :refer [map-vals]])
  (:import [java.net.http HttpClient HttpRequest HttpClient$Version HttpResponse$BodyHandlers]
           [java.util.concurrent LinkedBlockingQueue Executors]
           [java.net URI]
           [java.io IOException]
           [java.net SocketException]))

(declare request)

(def default-http-client-executor
  (delay
    (Executors/newFixedThreadPool
     (* 2 (.availableProcessors (Runtime/getRuntime))))))

(defn new-http-client
  []
  (-> (HttpClient/newBuilder)
      (.version HttpClient$Version/HTTP_2)
      (.executor @default-http-client-executor)
      (.build)))

(def default-http-client
  (atom (delay (new-http-client))))

(defn http-client
  [client]
  (or client @@default-http-client))

(defn reset-default-client!
  []
  (reset! default-http-client (delay (new-http-client))))

(defn retry-request
  [^Throwable e {:keys [connection-reset-retries] :as args}]
  (if (pos? connection-reset-retries)
    (do (reset-default-client!)
        (request args))
    (throw e)))

(defn request
  [{:keys [url headers method client connection-reset-retries]
    :or {method :get
         connection-reset-retries 1}
    :as args}]
  (try (let [^HttpClient client (http-client client)
             request (cond-> (HttpRequest/newBuilder)
                       true (.uri (URI/create url))
                       true (as-> %
                                (condp = method
                                  :get (.GET %)
                                  :post (.POST %)
                                  :put (.PUT %)
                                  :options (.OPTIONS %)))
                       (seq headers) (.headers (into-array (mapcat identity headers)))
                       true (.build))
             response (.send client request (HttpResponse$BodyHandlers/ofByteArray))]
         {:status (.statusCode response)
          :body (.body response)
          :headers (-> (->> (.headers response)
                            .map
                            (into {})
                            (map-vals first))
                       (dissoc ":status"))})
       (catch java.io.IOException e
         (->> (dec connection-reset-retries)
              (assoc args :connection-reset-retries)
              (retry-request e)))
       (catch java.net.SocketException e
         (->> (dec connection-reset-retries)
              (assoc args :connection-reset-retries)
              (retry-request e)))))
