;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.netty.http.server
  (:require [clojure.string :as st]
            [com.brunobonacci.mulog :as u])
  (:import [io.netty.channel
            ChannelDuplexHandler
            ChannelPipeline
            ChannelHandler
            ChannelHandlerContext]
           [java.util.concurrent ExecutorService]
           [io.netty.util ReferenceCounted]
           [io.netty.handler.codec.http
            DefaultFullHttpResponse
            HttpVersion
            HttpResponseStatus
            HttpRequest
            HttpServerCodec]
           [io.netty.handler.codec.http.websocketx
            WebSocketServerHandshakerFactory
            TextWebSocketFrame
            CloseWebSocketFrame]))

(defn websocket-request?
  [req]
  (boolean
   (and (= "websocket" (st/lower-case (str (get-in req [:headers "upgrade"]))))
        (= "upgrade" (st/lower-case (str (get-in req [:headers "connection"])))))))

(defn http1-request
  [^ChannelHandlerContext ctx ^HttpRequest msg]
  (let [headers (->> (.headers msg)
                     .entries
                     (map (fn [[k v]]
                            [(st/lower-case (str k)) (str v)]))
                     (into {}))
        path (.uri msg)
        [uri query-string] (st/split path #"\?")]
    {:headers headers
     :uri uri
     :path path
     :protocol-version (-> (.protocolVersion msg)
                           .text
                           st/lower-case)
     :scheme (if (websocket-request? {:headers headers})
               :wss
               :https)
     :query-string query-string
     :request-method (-> (.method msg)
                         .name
                         st/lower-case
                         keyword)
     :query-params (when query-string
                     (->> (st/split query-string #"&")
                          (map #(st/split % #"\="))
                          (into {})))}))

(defn configure-websocket-handlers
  [^ChannelHandlerContext ctx ^ExecutorService exec-service {:keys [on-open on-close on-text-message]}]
  (on-open {:send #(.writeAndFlush (.channel ctx) (TextWebSocketFrame. %))
            :close (fn [] (.close (.channel ctx)))})
  (doto (.pipeline ctx)
    (.remove "request-handler")
    (.addLast "frame-handler"
              (proxy [ChannelDuplexHandler] []
                (channelRead [^ChannelHandlerContext ctx msg]
                  (let [handler (cond
                                  (instance? TextWebSocketFrame msg)
                                  (let [text (.text ^TextWebSocketFrame msg)]
                                    #(on-text-message text))

                                  (instance? CloseWebSocketFrame msg)
                                  on-close

                                  :else nil)]
                    (try (when handler
                           (.submit exec-service
                                    (reify Runnable
                                      (run [_]
                                        (try
                                          (handler)
                                          (catch Exception e
                                            (u/log ::websocket-handler-config :exception e)))))))
                         (finally
                           (when (instance? ReferenceCounted msg)
                             (.release ^ReferenceCounted msg))))))))))

(defn handle-websocket-handshake
  [^ChannelHandlerContext ctx ^HttpRequest req]
  (let [websocket-url (str "wss://"
                           (.get (.headers req) "Host")
                           (.uri req))
        ws-factory (WebSocketServerHandshakerFactory. websocket-url nil true)
        handshaker (.newHandshaker ws-factory req)]
    (if (not handshaker)
      (WebSocketServerHandshakerFactory/sendUnsupportedVersionResponse (.channel ctx))
      (.handshake handshaker (.channel ctx) req))))

(defn ^ChannelHandler http1-request-handler
  [^ExecutorService exec-service handler]
  (proxy [ChannelDuplexHandler] []
    (channelRead [^ChannelHandlerContext ctx msg]
      (if (instance? HttpRequest msg)
        (let [^HttpRequest msg msg
              req (http1-request ctx msg)]
          (or (when (websocket-request? req)
                (let [response (handler req)]
                  (when (= 101 (:status response))
                    (let [{:keys [handlers]} (meta response)]
                      (configure-websocket-handlers ctx exec-service handlers)
                      (handle-websocket-handshake ctx msg))
                    true)))
              (do (.writeAndFlush
                   ctx (DefaultFullHttpResponse.
                        HttpVersion/HTTP_1_1
                        HttpResponseStatus/HTTP_VERSION_NOT_SUPPORTED))
                  (.close (.channel ctx)))))
        (proxy-super channelRead ctx msg)))))

(defn configure-http1-request-handler
  [^ChannelPipeline pipeline ^ExecutorService exec-service handler]
  (doto pipeline
    (.addLast "request-handler" (http1-request-handler exec-service handler))))

(defn configure-http1-pipeline
  [^ChannelPipeline pipeline ^ExecutorService exec-service handler]
  (doto pipeline
    (.addLast "frame-codec" (HttpServerCodec.))
    (configure-http1-request-handler exec-service handler)))
