;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.http.server
  (:require [via.netty :as netty]
            [utilis.fn :refer [fsafe]]
            [integrant.core :as ig]
            [clojure.set :as set]
            [metrics.meters :as meters]
            [metrics.counters :as counters]
            [metrics.timers :as timers]
            [clojure.java.io :as io]
            [clojure.set :refer [rename-keys]])
  (:import [io.netty.handler.ssl
            ApplicationProtocolConfig
            ApplicationProtocolConfig$Protocol
            ApplicationProtocolConfig$SelectorFailureBehavior
            ApplicationProtocolConfig$SelectedListenerFailureBehavior
            ApplicationProtocolNames
            SslContextBuilder
            SslProvider
            ClientAuth
            SupportedCipherSuiteFilter]
           [io.netty.handler.ssl.util SelfSignedCertificate]
           [io.netty.handler.codec.http2 Http2SecurityUtil]
           [java.net InetSocketAddress]
           [java.io Closeable]))

(defmethod ig/init-key :via.http/server
  [_ opts]
  (let [opts (rename-keys opts {:port :http-port
                                :host :http-host})
        ring-handler (atom (delay (:ring-handler opts)))
        {:keys [http-host http-port tls]
         :or {http-host "localhost"}} opts
        metric-keys (atom #{})
        metrics {:keys metric-keys
                 :static {:via.http.requests/meter (meters/meter ["via.http" "meter" "requests"])
                          :via.http.requests/timer (timers/timer ["via.http" "timer" "requests"])}
                 :dynamic {:via.http.requests.uri/timer (fn [uri]
                                                          (let [title ["via.http" "timer" (str "requests.uri" uri)]]
                                                            (swap! metric-keys conj [:via.http.requests.uri/timer title])
                                                            (timers/timer title)))}}]
    (reset! metric-keys
            #{[:via.http.requests/meter ["via.http" "meter" "requests"]]
              [:via.http.requests/timer ["via.http" "timer" "requests"]]})
    (when (not http-port)
      (throw (ex-info "An http-port must be provided to start :via.http/server"
                      {:http-host http-host
                       :http-port http-port})))
    {:ring-handler ring-handler
     :metrics metrics
     :http-server (netty/start-server
                   (merge
                    (select-keys opts [:leak-detector-level
                                       :initial-window-size
                                       :max-frame-size
                                       :max-concurrent-streams
                                       :max-header-list-size
                                       :push-enabled])
                    {:handler (fn [request]
                                (meters/mark! (-> metrics :static :via.http.requests/meter))
                                (->> ((fsafe @@ring-handler) request)
                                     (timers/time! ((:via.http.requests.uri/timer (:dynamic metrics)) (str (:uri request))))
                                     (timers/time! (:via.http.requests/timer (:static metrics)))))
                     :socket-address (InetSocketAddress. ^String http-host ^int http-port)
                     :ssl-context (-> (if tls
                                        (cond-> (SslContextBuilder/forServer
                                                 (io/file (get-in tls [:server :cert]))
                                                 (io/file (get-in tls [:server :key])))
                                          (get-in tls [:client :ca])
                                          (-> (.clientAuth (ClientAuth/REQUIRE))
                                              (.trustManager (io/file (get-in tls [:client :ca])))))
                                        (let [ssc (SelfSignedCertificate.)]
                                          (SslContextBuilder/forServer
                                           (.certificate ssc)
                                           (.privateKey ssc))))
                                      (.sslProvider SslProvider/JDK)
                                      (.ciphers Http2SecurityUtil/CIPHERS SupportedCipherSuiteFilter/INSTANCE)
                                      (.applicationProtocolConfig
                                       (ApplicationProtocolConfig.
                                        ApplicationProtocolConfig$Protocol/ALPN
                                        ApplicationProtocolConfig$SelectorFailureBehavior/NO_ADVERTISE
                                        ApplicationProtocolConfig$SelectedListenerFailureBehavior/ACCEPT
                                        (into-array [ApplicationProtocolNames/HTTP_2])))
                                      .build)}))}))

(defmethod ig/halt-key! :via.http/server
  [_ {:keys [http-server]}]
  (when http-server
    (.close ^Closeable http-server)))

(defmethod ig/suspend-key! :via.http/server
  [_ {:keys [ring-handler]}]
  (reset! ring-handler (promise)))

(defmethod ig/resume-key :via.http/server
  [key opts old-opts old-impl]
  (if (= (dissoc opts :ring-handler) (dissoc old-opts :ring-handler))
    (do (deliver @(:ring-handler old-impl) (:ring-handler opts))
        old-impl)
    (do (ig/halt-key! key old-impl)
        (ig/init-key key opts))))
