(ns via.netty.http2.client
  (:require [utilis.map :refer [map-vals]])
  (:import [java.net.http HttpClient HttpRequest HttpClient$Version HttpResponse$BodyHandlers]
           [java.util.concurrent LinkedBlockingQueue Executors]
           [java.net URI]))

(def default-http-client-executor
  (delay
    (Executors/newFixedThreadPool
     (* 2 (.availableProcessors (Runtime/getRuntime))))))

(def default-http-client
  (delay
    (-> (HttpClient/newBuilder)
        (.version HttpClient$Version/HTTP_2)
        (.executor @default-http-client-executor)
        (.build))))

(defn request
  [{:keys [url headers method client]
    :or {method :get}}]
  (let [^HttpClient client (or client @default-http-client)
        request (cond-> (HttpRequest/newBuilder)
                  true (.uri (URI/create url))
                  true (as-> %
                           (condp = method
                             :get (.GET %)
                             :post (.POST %)
                             :put (.PUT %)
                             :options (.OPTIONS %)))
                  (seq headers) (.headers (into-array (mapcat identity headers)))
                  true (.build))
        response (.send client request (HttpResponse$BodyHandlers/ofByteArray))]
    {:status (.statusCode response)
     :body (.body response)
     :headers (-> (->> (.headers response)
                       .map
                       (into {})
                       (map-vals first))
                  (dissoc ":status"))}))
