;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.core
  (:require [via.endpoint :as via]
            [via.events :as ve]
            [via.subs :as vs]
            [signum.subs :as ss]
            [signum.events :as se]
            [signum.fx :as sfx]
            [utilis.map :refer [compact]]))

(defn subscribe
  ([query]
   (let [endpoint (via/first-endpoint)]
     (subscribe endpoint (via/first-peer endpoint) query nil)))
  ([query default]
   (let [endpoint (via/first-endpoint)]
     (subscribe endpoint (via/first-peer endpoint) query default)))
  ([endpoint peer-id query]
   (subscribe endpoint peer-id query nil))
  ([endpoint peer-id query default]
   (if (ss/sub? (first query))
     (ss/subscribe query)
     (vs/subscribe endpoint peer-id query default))))

(defn dispatch
  "Dispatch `event` to `peer-id` through `endpoint`."
  ([event]
   (let [endpoint (via/first-endpoint)]
     (dispatch endpoint (via/first-peer endpoint) event)))
  ([peer-id event]
   (let [endpoint (via/first-endpoint)]
     (dispatch endpoint peer-id event)))
  ([endpoint peer-id event]
   (if (se/event? (first event))
     (se/dispatch nil event)
     (ve/dispatch endpoint peer-id event))))

(defn invoke
  "Invoke the handler for `event` on `peer-id` through `endpoint`,
  with a response from the peer expected, which can be handled using
  promise syntax."
  ([event]
   (let [endpoint (via/first-endpoint)]
     (invoke endpoint (via/first-peer endpoint) event nil)))
  ([event options]
   (let [endpoint (via/first-endpoint)]
     (invoke endpoint (via/first-peer endpoint) event options)))
  ([endpoint peer-id event]
   (invoke endpoint peer-id event nil))
  ([endpoint peer-id event options]
   (ve/invoke endpoint peer-id event options)))

(sfx/reg-fx
 :via/dispatch
 (fn [_ event-or-map]
   (cond
     (sequential? event-or-map)
     (dispatch (vec event-or-map))

     (map? event-or-map)
     (let [{:keys [peer-id endpoint event]} event-or-map]
       (cond
         (and endpoint peer-id)
         (dispatch endpoint peer-id event)

         peer-id
         (dispatch peer-id event)

         :else (dispatch event)))
     :else (throw (ex-info "Argument to :via/dispatch must be an event vector or a map of options."
                           {:event-or-map event-or-map})))))

(sfx/reg-fx
 :via/invoke
 (fn [_ {:keys [event endpoint peer-id
               timeout on-success on-failure on-timeout]}]
   (let [endpoint (or endpoint (via/first-endpoint))]
     (invoke endpoint
             (or peer-id (via/first-peer endpoint))
             event
             (compact
              {:timeout timeout
               :on-success on-success
               :on-failure on-failure
               :on-timeout on-timeout})))))
