;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns via.netty.http.server
  (:require [utilis.map :refer [map-keys map-vals compact]]
            [byte-streams.core :as bs]
            [manifold.deferred :as d]
            [manifold.executor :as e]
            [manifold.stream :as s]
            [manifold.stream.core :as manifold]
            [clojure.string :as st])
  (:import [java.io IOException]
           [io.netty.bootstrap Bootstrap ServerBootstrap]
           [io.netty.buffer ByteBuf Unpooled]
           [io.netty.channel
            ChannelFutureListener
            ChannelPromise
            ChannelDuplexHandler
            Channel ChannelFuture ChannelOption
            ChannelPipeline EventLoopGroup
            DefaultChannelPipeline
            ChannelHandler FileRegion
            ChannelInboundHandler
            ChannelOutboundHandler
            ChannelHandlerContext
            ChannelInitializer]
           [java.util.concurrent
            ConcurrentHashMap
            CancellationException
            ScheduledFuture
            TimeUnit
            ThreadFactory
            Executors
            ExecutorService]
           [io.netty.channel.epoll Epoll EpollEventLoopGroup
            EpollServerSocketChannel
            EpollSocketChannel
            EpollDatagramChannel]
           [io.netty.util Attribute AttributeKey]
           [io.netty.handler.codec Headers]
           [io.netty.handler.codec.http
            DefaultFullHttpResponse
            HttpVersion
            HttpResponseStatus
            HttpRequest
            HttpServerCodec]
           [io.netty.handler.codec.http.websocketx
            WebSocketServerHandshakerFactory
            TextWebSocketFrame
            CloseWebSocketFrame
            WebSocketFrame
            WebSocket13FrameDecoder
            WebSocket13FrameEncoder
            WebSocketDecoderConfig]
           [io.netty.handler.codec.http.websocketx.extensions
            WebSocketServerExtension
            WebSocketExtensionData
            WebSocketExtensionDecoder
            WebSocketExtensionEncoder]
           [io.netty.handler.codec.http.websocketx.extensions.compression
            PerMessageDeflateServerExtensionHandshaker]
           [io.netty.handler.codec.http2
            DefaultHttp2DataFrame
            DefaultHttp2Headers
            DefaultHttp2HeadersFrame
            EmptyHttp2Headers
            Http2FrameCodecBuilder
            Http2HeadersFrame
            Http2DataFrame
            Http2FrameStream
            Http2Headers
            Http2Settings]
           [io.netty.channel.nio NioEventLoopGroup]
           [io.netty.channel.socket
            ServerSocketChannel
            SocketChannel]
           [io.netty.channel.socket.nio
            NioServerSocketChannel
            NioSocketChannel
            NioDatagramChannel]
           [io.netty.handler.ssl
            SslContext
            SslContextBuilder
            SslHandler
            SslHandshakeCompletionEvent
            ApplicationProtocolNegotiationHandler
            ApplicationProtocolNames]
           [java.net URI SocketAddress InetSocketAddress]
           [java.nio ByteBuffer]))

(defn websocket-request?
  [req]
  (boolean
   (and (= "websocket" (st/lower-case (str (get-in req [:headers "upgrade"]))))
        (= "upgrade" (st/lower-case (str (get-in req [:headers "connection"])))))))

(defn http1-request
  [^HttpRequest msg]
  (let [headers (->> (.headers msg)
                     .entries
                     (map (fn [[k v]]
                            [(st/lower-case (str k)) (str v)]))
                     (into {}))
        path (.uri msg)
        [uri query-string] (st/split path #"\?")]
    {:headers headers
     :uri uri
     :path path
     :protocol-version (st/lower-case (.text (.protocolVersion msg)))
     :scheme (keyword (first (st/split (get headers "origin") #"://")))
     :query-string query-string
     :request-method (-> (.method msg)
                         .name
                         st/lower-case
                         keyword)
     :query-params (when query-string
                     (->> (st/split query-string #"&")
                          (map #(st/split % #"\="))
                          (into {})))}))

(defn configure-websocket-handlers
  [^ChannelHandlerContext ctx {:keys [on-open on-close on-text-message]}]
  (on-open {:send #(.writeAndFlush (.channel ctx) (TextWebSocketFrame. %))
            :close (fn [] (.close (.channel ctx)))})
  (doto (.pipeline ctx)
    (.remove "request-handler")
    (.addLast "frame-handler"
              (proxy [ChannelDuplexHandler] []
                (channelRead [^ChannelHandlerContext ctx msg]
                  (cond
                    (instance? TextWebSocketFrame msg)
                    (on-text-message (.text ^TextWebSocketFrame  msg))

                    (instance? CloseWebSocketFrame msg)
                    (on-close)

                    :else nil))))))

(defn handle-websocket-handshake
  [^ChannelHandlerContext ctx ^HttpRequest req]
  (let [websocket-url (str "wss://"
                           (.get (.headers req) "Host")
                           (.uri req))
        ws-factory (WebSocketServerHandshakerFactory. websocket-url nil true)
        handshaker (.newHandshaker ws-factory req)]
    (if (not handshaker)
      (WebSocketServerHandshakerFactory/sendUnsupportedVersionResponse (.channel ctx))
      (.handshake handshaker (.channel ctx) req))))

(defn ^ChannelHandler http1-request-handler
  [^ExecutorService exec-service handler]
  (proxy [ChannelDuplexHandler] []
    (channelRead [^ChannelHandlerContext ctx msg]
      (if (instance? HttpRequest msg)
        (let [^HttpRequest msg msg
              req (http1-request msg)]
          (or (when (websocket-request? req)
                (let [response (handler req)]
                  (when (= 101 (:status response))
                    (let [{:keys [handlers]} (meta response)]
                      (configure-websocket-handlers ctx handlers)
                      (handle-websocket-handshake ctx msg))
                    true)))
              (do (.writeAndFlush
                   ctx (DefaultFullHttpResponse.
                        HttpVersion/HTTP_1_1
                        HttpResponseStatus/HTTP_VERSION_NOT_SUPPORTED))
                  (.close (.channel ctx)))))
        (proxy-super channelRead ctx msg)))))

(defn configure-http1-request-handler
  [^ChannelPipeline pipeline ^ExecutorService exec-service handler]
  (doto pipeline
    (.addLast "request-handler" (http1-request-handler exec-service handler))))

(defn configure-http1-pipeline
  [^ChannelPipeline pipeline ^ExecutorService exec-service handler]
  (doto pipeline
    (.addLast "frame-codec" (HttpServerCodec.))
    (configure-http1-request-handler exec-service handler)))
