(ns via.util.request
  (:require [clojure.string :as st]
            [utilis.map :refer [compact]]
            [utilis.types.number :refer [string->double
                                         string->long]])
  #?(:clj (:import [java.net URLEncoder URLDecoder])))

(defn parse-via-value-string
  [s]
  (re-find #"^via/(.*)=(.*)$" s))

(defn url-encode
  [^String string]
  #?(:clj (URLEncoder/encode string "UTF-8")
     :cljs (js/encodeURIComponent string)))

(defn url-decode
  [^String string]
  #?(:clj (URLDecoder/decode string "UTF-8")
     :cljs (js/decodeURIComponent string)))

(defn encode-query-param
  [value]
  (cond
    (string? value) value
    (keyword? value) (str "via/keyword=" (name value))
    (integer? value) (str "via/int=" value)
    (number? value) (str "via/double=" value)
    :else (str value)))

(defn decode-query-param
  [value]
  (if-let [[_ type value] (parse-via-value-string value)]
    (condp = type
      "keyword" (keyword value)
      "int" (string->long value)
      "double" (string->double value)
      (throw (ex-info "Unable to parse query parameter" {:value value :type type})))
    value))

(defn parse-query-string
  [query-string & {:keys [url-decode-values] :or {url-decode-values true}}]
  (when query-string
    (->> (st/split query-string #"&")
         (reduce (fn [params kv-string]
                   (let [[key value] (st/split kv-string #"=")]
                     (->> (cond-> value
                            url-decode-values (url-decode))
                          decode-query-param
                          (assoc params (keyword key)))))
                 {})
         compact
         not-empty)))
