;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns ventus.hooks
  (:require
   [helix.hooks :refer [use-effect use-layout-effect use-ref use-state]]
   [signum.core :refer [hot? subscribe]]
   [utilis.js :as j]
   [utilis.timer :refer [run-after]]))

(def ^:private signal-dispose-delay 5000)

(defn use-signal
  ([signal] (use-signal signal ::suspend))
  ([signal default]
   (let [[value set-value] (use-state (when (and signal (hot? signal)) @signal))]
     (use-effect
      [signal]
      (when signal
        (let [watch-key (str "use-signal/effect-" (random-uuid))]
          (add-watch signal watch-key
                     (fn [_ _ _ new-value]
                       (set-value new-value)))
          (fn []
            (run-after
             #(remove-watch signal watch-key)
             signal-dispose-delay)))))
     (if (and signal (hot? signal))
       value
       (if (and signal (= ::suspend default))
         (let [watch-key (str "use-signal/promise-" (random-uuid))
               pending (js/Promise.
                        (fn [resolve _reject]
                          (add-watch signal watch-key
                                     (fn [_ _ _ new-value]
                                       (resolve new-value)))))]
           (j/call pending :finally (fn [_]
                                      (run-after
                                       #(remove-watch signal watch-key)
                                       signal-dispose-delay)))
           (throw pending))
         default)))))

(defn use-sub
  ([query-v]
   (use-signal (subscribe query-v)))
  ([query-v default]
   (use-signal (subscribe query-v) default)))

(defn use-long-press
  [element-ref on-press & {:keys [threshold on-release] :or {threshold 400}}]
  (let [timer (use-ref nil)]
    (use-effect
     [element-ref]
     (let [element @element-ref
           clear (fn [] (when @timer
                         (js/clearTimeout @timer)
                         (reset! timer nil)))
           down (fn [e]
                  (when-not @timer
                    (reset! timer (js/setTimeout
                                   (fn []
                                     (clear)
                                     (on-press e)) threshold))))
           up (fn [e]
                (clear)
                (when on-release (on-release e)))]
       (when element
         (j/call element :addEventListener "pointerdown" down)
         (j/call element :addEventListener "pointerup" up)
         (fn []
           (clear)
           (j/call element :removeEventListener "pointerdown" down)
           (j/call element :removeEventListener "pointerup" up)))))))

(defn use-media-query
  [query]
  (let [[result set-result] (use-state false)]
    (use-effect
     [query]
     (let [result (js/matchMedia query)
           on-change (fn [e]
                       (set-result (j/get e :matches)))]
       (j/call result :addEventListener "change" on-change)
       (on-change result)
       (fn []
         (j/call result :removeEventListener "change" on-change))))
    result))

(def ^:private stashed-time (atom {}))

(defn use-synchronized-animation
  [animation-name]
  (let [ref (use-ref nil)]
    (use-layout-effect
     [animation-name]
     (let [animations (->> (j/call js/document :getAnimations)
                           (filter #(= animation-name (j/get % :animationName))))
           current (first (filter #(= @ref (j/get-in % [:effect :target])) animations))
           first? (= current (first animations))]
       (if first?
         (when-let [time (get @stashed-time animation-name)]
           (j/assoc! current :currentTime time))
         (->> (j/get (first animations) :currentTime)
              (j/assoc! current :currentTime)))
       (fn []
         (when first?
           (->> (j/get current :currentTime)
                (swap! stashed-time assoc animation-name))))))
    ref))





(defn use-escape
  [handler]
  (use-effect
   :once
   (let [h (fn [e]
             (when (= 27 (j/get e :keyCode))
               (handler)))]
     (js/window.addEventListener "keydown" h)
     #(js/window.removeEventListener "keydown" h))))

(defn use-layout
  [element]
  (let [[layout set-layout] (use-state nil)]
    (use-effect
     [element]
     (when element
       (let [observer (js/ResizeObserver.
                       (fn [entries]
                         (let [entry (j/get-in entries [0 :contentRect])]
                           (set-layout
                            {:width (j/get entry :width)
                             :height (j/get entry :height)
                             :x (j/get entry :x)
                             :y (j/get entry :y)}))))]
         (j/call observer :observe element)
         (fn []
           (j/call observer :unobserve element)))))
    layout))

(defn use-mutations
  [element]
  (let [[mutations set-mutations] (use-state nil)]
    (use-effect
     [element]
     (when element
       (let [observer (js/MutationObserver.
                       (fn [mutation-list _observer]
                         (set-mutations mutation-list)))]
         (j/call observer :observe element
                 #js {:attributes true
                      :childList true
                      :subtree true})
         (fn [] (j/call observer :disconnect)))))
    mutations))
