;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns ventus.cookie
  (:refer-clojure :exclude [get empty? count keys])
  (:require
   [clojure.string :as st]
   [tempus.duration :as td])
  (:import
   goog.net.Cookies))

(defonce ^:private cookies (Cookies. js/document))

(defn enabled?
  []
  (.isEnabled cookies))

(defn empty?
  []
  (.isEmpty cookies))

(defn count
  []
  (.count cookies))

(defn keys
  []
  (vec (.getKeys cookies)))

(defn values
  []
  (vec (.getValues cookies)))

(defn get
  ([key] (get key nil))
  ([key default]
   (.get cookies (name key) default)))

(defn set!
  [key value & {:keys [max-age path domain secure same-site http-only]
                :or {max-age nil ; session cookie
                     secure true
                     same-site :strict
                     http-only true}}]
  (if (and (.isValidName cookies (name key)) (.isValidValue cookies value))
    (.set cookies (name key) (name value)
          #js {"maxAge" (if max-age (td/into :seconds max-age) -1)
               "path" path
               "domain" domain
               "secure" secure
               "sameSite" (-> same-site name st/upper-case)
               "httpOnly" http-only})
    (throw (ex-info ":ventus/cookie invalid" {:key key :value value}))))

(defn delete!
  [key]
  (.remove cookies (name key)))
