;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns ventus.hooks
  (:require
   ["react" :as react]
   [helix.hooks :refer [use-effect use-ref use-state]]
   [signum.signal :refer [hot?] :as s]
   [utilis.js :as j]
   [utilis.timer :refer [run-after]]))

(def ^:private signal-dispose-delay 5000)
(defonce ^:private pending-signals (atom {}))

(defn use-signal
  ([signal] (use-signal signal ::suspend))
  ([signal default]
   (let [[value set-value] (use-state (when (hot? signal) @signal))]
     (use-effect
      [signal]
      (let [watch-key (str "use-signal/effect" (random-uuid))]
        (add-watch
         signal watch-key
         (fn [_ _ _ new-value]
           (set-value new-value)))
        (when-let [pending (get @pending-signals signal)]
          (swap! pending-signals dissoc signal)
          (remove-watch signal (:watch-key pending)))
        (fn []
          (run-after
           #(remove-watch signal watch-key)
           signal-dispose-delay))))
     (if (hot? signal)
       value
       (if (= ::suspend default)
         (react/use
          (if-let [pending (get-in @pending-signals [signal :promise])]
            pending
            (let [watch-key (str "use-signal/promise" (random-uuid))
                  pending (js/Promise.
                           (fn [resolve _reject]
                             (add-watch signal watch-key
                                        (fn [_ _ _ new-value] (resolve new-value)))))]
              (swap! pending-signals assoc signal {:promise pending :watch-key watch-key})
              pending)))
         default)))))

(defn use-escape
  [handler]
  (use-effect
   :once
   (let [h (fn [e]
             (when (= 27 (j/get e :keyCode))
               (handler)))]
     (js/window.addEventListener "keydown" h)
     #(js/window.removeEventListener "keydown" h))))

(defn use-layout
  [element]
  (let [[layout set-layout] (use-state nil)]
    (use-effect
     [element]
     (when element
       (let [observer (js/ResizeObserver.
                       (fn [entries]
                         (let [entry (j/get-in entries [0 :contentRect])]
                           (set-layout
                            {:width (j/get entry :width)
                             :height (j/get entry :height)
                             :x (j/get entry :x)
                             :y (j/get entry :y)}))))]
         (j/call observer :observe element)
         (fn []
           (j/call observer :unobserve element)))))
    layout))

(defn use-mutations
  [element]
  (let [[mutations set-mutations] (use-state nil)]
    (use-effect
     [element]
     (when element
       (let [observer (js/MutationObserver.
                       (fn [mutation-list _observer]
                         (set-mutations mutation-list)))]
         (j/call observer :observe element
                 #js {:attributes true
                      :childList true
                      :subtree true})
         (fn [] (j/call observer :disconnect)))))
    mutations))

(defn use-pin-scroll-to-bottom
  [scroll-el scroll-content-el]
  (let [layout (use-layout scroll-content-el)
        mutations (use-mutations scroll-content-el)
        scrolled? (use-ref false)]
    (use-effect
     [scroll-el]
     (when scroll-el
       (let [listener (fn listener [] (reset! scrolled? true))]
         (j/call scroll-el :addEventListener "scroll" listener #js {:once true})
         (fn [] (j/call scroll-el :removeEventListener "scroll" listener)))))
    (use-effect
     [scroll-el scroll-content-el layout mutations]
     (when scroll-el
       (js/requestAnimationFrame
        (fn []
          (let [children (j/get scroll-content-el :children)
                n (j/get children :length)
                last-child-height (+ 8 (or (when (pos? n)
                                             (j/get-in children [(dec n) :offsetHeight]))
                                           0))
                scroll-bottom (- (j/get-in scroll-el [:scrollHeight])
                                 (j/get scroll-el :scrollTop)
                                 (-> scroll-el
                                     (j/call :getBoundingClientRect)
                                     (j/get :height))
                                 last-child-height)]
            (when (or (not @scrolled?) (<= scroll-bottom 100))
              (j/assoc! scroll-el :scrollTop
                        (j/get-in scroll-el [:scrollHeight]))))))))))
