;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns ventus.hooks
  (:require [signum.signal :refer [hot?] :as s]
            [helix.hooks :refer [use-state use-effect use-ref]]
            [utilis.timer :refer [run-after]]
            [utilis.js :as j]))

(def ^:private signal-dispose-delay 5000)

(defn use-signal
  [signal]
  (let [[value set-value] (use-state (when (hot? signal) @signal))]
    (use-effect
     [signal]
     ;; Effect will not run if the promise is thrown
     (let [watch-key (random-uuid)]
       (add-watch
        signal watch-key
        (fn [_ _ _ new-value]
          (set-value new-value)))
       #(run-after (fn [] (remove-watch signal watch-key))
                   signal-dispose-delay)))
    (if (hot? signal)
      value
      (throw
       (js/Promise.
        (fn [resolve reject]
          (let [watch-key (str "p-" (random-uuid))]
            (add-watch
             signal watch-key
             (fn [_ _ _ new-value]
               ((if (instance? js/Error new-value) reject resolve) new-value)
               (run-after (fn [] (remove-watch signal watch-key))
                          signal-dispose-delay))))))))))

(defn use-escape
  [handler]
  (use-effect
   :once
   (let [h (fn [e]
             (when (= 27 (j/get e :keyCode))
               (handler)))]
     (js/window.addEventListener "keydown" h)
     #(js/window.removeEventListener "keydown" h))))

(defn use-layout
  [element]
  (let [[layout set-layout] (use-state nil)]
    (use-effect
     [element]
     (when element
       (let [observer (js/ResizeObserver.
                       (fn [entries]
                         (let [entry (j/get-in entries [0 :contentRect])]
                           (set-layout
                            {:width (j/get entry :width)
                             :height (j/get entry :height)
                             :x (j/get entry :x)
                             :y (j/get entry :y)}))))]
         (j/call observer :observe element)
         (fn []
           (j/call observer :unobserve element)))))
    layout))

(defn use-mutations
  [element]
  (let [[mutations set-mutations] (use-state nil)]
    (use-effect
     [element]
     (when element
       (let [observer (js/MutationObserver.
                       (fn [mutation-list _observer]
                         (set-mutations mutation-list)))]
         (j/call observer :observe element
                 #js {:attributes true
                      :childList true
                      :subtree true})
         (fn [] (j/call observer :disconnect)))))
    mutations))

(defn use-pin-scroll-to-bottom
  [scroll-el scroll-content-el]
  (let [layout (use-layout scroll-content-el)
        mutations (use-mutations scroll-content-el)
        scrolled? (use-ref false)]
    (use-effect
     [scroll-el]
     (when scroll-el
       (let [listener (fn listener [] (reset! scrolled? true))]
         (j/call scroll-el :addEventListener "scroll" listener #js {:once true})
         (fn [] (j/call scroll-el :removeEventListener "scroll" listener)))))
    (use-effect
     [scroll-el scroll-content-el layout mutations]
     (when scroll-el
       (js/requestAnimationFrame
        (fn []
          (let [children (j/get scroll-content-el :children)
                n (j/get children :length)
                last-child-height (+ 8 (or (when (pos? n)
                                             (j/get-in children [(dec n) :offsetHeight]))
                                           0))
                scroll-bottom (- (j/get-in scroll-el [:scrollHeight])
                                 (j/get scroll-el :scrollTop)
                                 (-> scroll-el
                                     (j/call :getBoundingClientRect)
                                     (j/get :height))
                                 last-child-height)]
            (when (or (not @scrolled?) (<= scroll-bottom 100))
              (j/assoc! scroll-el :scrollTop
                        (j/get-in scroll-el [:scrollHeight]))))))))))
