;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.websocket
  (:require
   [clojure.string :as st]
   [fluxus.flow :as f]
   [fluxus.promise :as p]
   [utilis.js :as j]
   [vectio.http :as http]))

(set! *warn-on-infer* true)

(defn origin
  ([] (origin nil))
  ([path]
   (st/replace (http/origin path)  #"^http" "ws")))

(defn connect
  [{:keys [socket-fn address on-close on-error]}]
  (let [[client-stream internal] (f/entangled)
        connection (p/promise)
        status (fn [e]
                 (let [reason (or (j/get e :reason)
                                  (j/get e :message))]
                   (cond-> {:reason reason
                            :code (.-code e)
                            :event e}
                     (boolean (re-find #"(?i)403" (str reason)))
                     (assoc :fatal? true))))
        ^js/WebSocket
        socket (if socket-fn
                 (socket-fn)
                 (js/WebSocket. address))]
    (set! (.-onopen socket)
          (fn [_e]
            (p/resolve! connection client-stream)))
    (set! (.-onmessage socket)
          (fn [e]
            (f/put! internal (.-data e))))
    (set! (.-onclose socket)
          (fn [e]
            (f/close! internal)
            (when on-close
              (on-close (status e)))))
    (set! (.-onerror socket)
          (fn [e]
            (when on-error
              (on-error e))
            (f/close! internal)
            (let [status (status e)]
              (when-not (p/realized? connection)
                (p/reject! connection status)))))
    (f/on-close client-stream (fn [_] (.close socket)))
    (f/consume (fn [message] (.send socket ^js/String message)) internal)
    connection))

(defn disconnect
  [connection]
  (when (p/realized? connection)
    (f/close! @connection)))
