;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.jetty.websocket
  (:require
   [spectator.log :as log]
   [tempus.duration :as td]
   [utilis.timer :as timer])
  (:import
   [java.nio ByteBuffer]
   [java.time Duration]
   [org.eclipse.jetty.server Request]
   [org.eclipse.jetty.websocket.api Callback Session Session$Listener$AutoDemanding]
   [org.eclipse.jetty.websocket.server ServerWebSocketContainer WebSocketCreator]))

(declare websocket-creator)

(defn upgrade
  [{:keys [^Request request response callback]} handlers options]
  (let [container (ServerWebSocketContainer/get (.getContext request))
        creator (websocket-creator handlers options)
        upgrade (.upgrade container creator request response callback)]
    (log/trace [:vectio.jetty.websocket/timeout (.getIdleTimeout container)])
    (log/trace [:vectio.jetty.websocket/upgrade upgrade])
    {:status (if upgrade 200 400)}))


;;; Private

(deftype Listener [handlers keep-alive-index keep-alive-timer]
  Session$Listener$AutoDemanding
  (^void onWebSocketOpen [_ ^Session session]
   (log/trace [:vectio.jetty.websocket/on-open session])
   (.setIdleTimeout session (Duration/ofHours 24))
   (reset! keep-alive-timer
           (timer/run-every
            (fn []
              (let [sn (swap! keep-alive-index inc)]
                (log/trace [:vectio.jetty.websocket/ping sn])
                (try
                  (.sendPing session (.putLong (ByteBuffer/allocate 8) sn) Callback/NOOP)
                  (catch Throwable e
                    (log/error [:vectio.jetty.websocket/ping] e)))))
            (td/into :milliseconds (td/seconds 50))))
   ((:on-open handlers)
    {:send (fn [message]
             (try
               (if (string? message)
                 (.sendText session message Callback/NOOP)
                 (.sendBinary session message Callback/NOOP))
               (catch Throwable e
                 (log/error [:vectio.jetty.websocket/send] e))))
     :close (fn []
              (.close session))}))
  (^void onWebSocketClose [_ ^int status ^String reason]
   (log/trace [:vectio.jetty.websocket/on-close status reason])
   (when-let [t @keep-alive-timer]
     (timer/cancel t)
     (reset! keep-alive-timer nil))
   ((:on-close handlers) status reason))
  (^void onWebSocketText [_ ^String message]
   (log/trace [:vectio.jetty.websocket/on-text-message message])
   ((:on-text-message handlers) message))
  (^void onWebSocketBinary [_ ^ByteBuffer payload ^Callback cb]
   (log/trace [:vectio.jetty.websocket/on-binary-message])
   ((:on-binary-message handlers) payload))
  (^void onWebSocketPing [_ ^ByteBuffer bytebuffer]
   (log/trace [:vectio.jetty.websocket/on-ping bytebuffer]))
  (^void onWebSocketPong [_ ^ByteBuffer bytebuffer]
   (log/trace [:vectio.jetty.websocket/on-pong bytebuffer]))
  (^void onWebSocketError [_ ^Throwable e]
   ((:on-error handlers) e)))

(defn- websocket-creator
  [handlers _options]
  (reify WebSocketCreator
    (createWebSocket [_ _request _response _callback]
      (Listener. handlers (atom -1) (atom nil)))))
