;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.util.multipart
  (:require  [utilis.io :as io])
  (:import [java.net.http HttpRequest$BodyPublishers]
           [java.nio.charset StandardCharsets]
           [java.io File InputStream SequenceInputStream]
           [java.util Collections]
           [java.util.function Supplier]))

(declare boundary encode)

(defn request
  [request]
  (encode request (boundary)))

(defn- boundary
  "https://www.ietf.org/rfc/rfc2046.txt"
  []
  (->> (repeatedly #(rand-nth "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWQYZ_"))
       (take 32)
       (apply str "vectio-")))

(defn- content-disposition
  [{:keys [part-name name content file-name] :as part}]
  (str "Content-Disposition: form-data; "
       (format "name=\"%s\"" (or part-name name
                                 (throw (ex-info "missing :name or :part-name"
                                                 {:part part}))))
       (if (or (instance? File content)
               (instance? InputStream content))
         (format "; filename=\"%s\"" file-name)
         (throw (ex-info "missing :filename" {:part part})))))

(defn- content-type
  [{:keys [content content-type] :as part}]
  (str "Content-Type: "
       (cond
         content-type content-type
         (string? content) "text/plain; charset=UTF-8"
         (and (or (instance? File content)
                  (instance? InputStream content))
              (not content-type)) (throw (ex-info "missing :content-type"
                                                  {:part part}))
         :else (throw (ex-info "invalid part" {:part part})))))

(defn- content-transfer-encoding
  [{:keys [content]}]
  (if (string? content)
    "Content-Transfer-Encoding: 8bit"
    "Content-Transfer-Encoding: binary"))

(defn- encode-parts
  [parts boundary]
  (let [line-break "\r\n"
        payload-end (.getBytes (str "--" boundary "--" line-break)
                               StandardCharsets/UTF_8)]
    (concat
     (mapcat (fn [part]
               [(let [header (str "--"
                                  boundary
                                  line-break
                                  (content-disposition part)
                                  line-break
                                  (content-type part)
                                  line-break
                                  (content-transfer-encoding part)
                                  line-break
                                  line-break)]
                  [(count header) (.getBytes header StandardCharsets/UTF_8)])
                [(if-let [length (:content-length part)]
                   length
                   (let [content (:content part)]
                     (cond
                       (or (bytes? content)
                           (string? content)) (count content)
                       (instance? File content) (.length ^java.io.File content)
                       :else (throw (ex-info ":content-length missing" {:part part})))
                     ))
                 (io/input-stream (:content part))]
                [(count line-break) (.getBytes line-break StandardCharsets/UTF_8)]])
             parts)
     [[(count payload-end) payload-end]])))

(defn body
  [{:keys [multipart]} boundary]
  (let [parts (encode-parts multipart boundary)]
    (HttpRequest$BodyPublishers/fromPublisher
     (HttpRequest$BodyPublishers/ofInputStream
      (reify Supplier (get [_]
                        (->> parts
                             (map (comp io/input-stream second))
                             Collections/enumeration
                             (SequenceInputStream.)))))
     (->> parts (map first) (reduce +)))))

(defn encode
  [request boundary]
  (-> request
      (dissoc :multipart)
      (assoc :body (body request boundary))
      (update :headers assoc :content-type (str "multipart/form-data; boundary=" boundary))))
