;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.http
  (:refer-clojure :exclude [get])
  (:require [vectio.netty.server :as server]
            [vectio.netty.http1.websocket :as http1-websocket]
            [vectio.util.fides-tls :as fides]
            [vectio.util.multipart :as multipart]
            [fluxus.promise :as p]
            [utilis.types.number :refer [string->long]]
            [clojure.string :as st]
            [integrant.core :as ig])
  (:import [io.netty.handler.ssl
            ApplicationProtocolConfig
            ApplicationProtocolConfig$Protocol
            ApplicationProtocolConfig$SelectorFailureBehavior
            ApplicationProtocolConfig$SelectedListenerFailureBehavior
            ApplicationProtocolNames
            SslContextBuilder
            SslProvider
            ClientAuth
            SupportedCipherSuiteFilter
            JdkSslContext]
           [io.netty.handler.codec.http2 Http2SecurityUtil]
           [java.net.http HttpClient HttpClient$Version
            HttpRequest HttpRequest$Builder HttpRequest$BodyPublishers HttpRequest$BodyPublisher
            HttpResponse HttpResponse$BodyHandlers]
           [javax.net.ssl SSLParameters]
           [java.net InetSocketAddress URI]
           [java.util.concurrent Executors]
           [java.io InputStream Closeable File]
           [java.util.function Function Supplier]
           [java.time Duration]))

(declare server ensure-http-client server-ssl-context client-ssl-context)

(defmethod ig/init-key :vectio.http/server
  [_ {:keys [host port ring-handler] :as opts}]
  (when (not port)
    (throw (ex-info "A port must be provided to start :vectio.http/server"
                    {:host host :port port})))
  {:ring-handler ring-handler
   :http-server (server opts)})

(defmethod ig/halt-key! :vectio.http/server
  [_ {:keys [http-server]}]
  (when http-server
    (.close ^Closeable http-server)))

(defmethod ig/suspend-key! :vectio.http/server
  [_ {:keys [ring-handler]}]
  (reset! ring-handler (promise)))

(defmethod ig/resume-key :vectio.http/server
  [key opts old-opts old-impl]
  (if (= (dissoc opts :ring-handler) (dissoc old-opts :ring-handler))
    (do (deliver @(:ring-handler old-impl) (:ring-handler opts))
        old-impl)
    (do (ig/halt-key! key old-impl)
        (ig/init-key key opts))))

(defn server
  [{:keys [host port
           ring-handler
           tls ssl-context]
    :as server-opts}]
  (server/start-server
   (merge
    (select-keys server-opts [:leak-detector-level
                              :initial-window-size
                              :websocket-max-frame-size
                              :max-frame-size
                              :max-flush-size
                              :max-concurrent-streams
                              :max-header-list-size
                              :push-enabled
                              :default-outbound-max-frame-size
                              :protocols])
    {:handler ring-handler
     :socket-address (InetSocketAddress. ^String host ^int port)
     :ssl-context (or ssl-context
                      (when (not-empty tls)
                        (server-ssl-context tls)))})))

(defn websocket-client
  [{:keys [host port tls path]
    :as websocket-args}]
  (http1-websocket/websocket-client-stream websocket-args))

(defn websocket-stream-response
  [request]
  (server/websocket-stream-response request))

(def ^:private client-executor
  (Executors/newThreadPerTaskExecutor
   (-> (Thread/ofVirtual)
       (.name "vectio.http.client-" 0)
       (.factory))))

(defn client
  (^HttpClient [] (client {}))
  (^HttpClient [{:keys [tls ssl-context timeout]}]
   (let [ssl-context (or ssl-context
                         (when (not-empty tls)
                           (client-ssl-context tls)))]
     (cond-> (-> (HttpClient/newBuilder)
                 (.executor client-executor))
       timeout (.connectTimeout (Duration/ofMillis timeout))
       ssl-context (.sslParameters (doto (SSLParameters.)
                                     (.setNeedClientAuth true)))
       ssl-context (.sslContext ssl-context)
       true .build))))

(declare send-request)

(defn get
  ([client url] (get client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :get))))

(defn post
  ([client url] (post client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :post))))

(defn put
  ([client url] (put client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :put))))

(defn patch
  ([client url] (patch client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :patch))))

(defn delete
  ([client url] (delete client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :delete))))

(defn head
  ([client url] (head client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :head))))

(defn options
  ([client url] (options client url nil))
  ([client url request]
   (send-request client url (assoc request :request-method :options))))


;;; Private

(defn- parse-response
  [^HttpResponse response]
  {:url (str (.uri response))
   :status (.statusCode response)
   :body (.body response)
   :headers (->> (.map (.headers response))
                 (map (fn [[k v]]
                        (let [v (cond-> (if (> (count v) 1) v (first v))
                                  (= "content-length" k) string->long)]
                          [(keyword k) v])))
                 (into {}))})

(defn- send-request
  [^HttpClient client url request]
  (let [request (cond-> request (:multipart request) multipart/request)
        {:keys [request-method headers body-as body timeout]
         :or {body-as :input-stream}} request
        ^HttpRequest$Builder builder (-> (HttpRequest/newBuilder)
                                         (.uri (URI. url))
                                         (.version HttpClient$Version/HTTP_2)
                                         (.method (st/upper-case (name request-method))
                                                  (cond
                                                    (instance? HttpRequest$BodyPublisher body) body
                                                    (nil? body) (HttpRequest$BodyPublishers/noBody)
                                                    (bytes? body) (HttpRequest$BodyPublishers/ofByteArray body)
                                                    (string? body) (HttpRequest$BodyPublishers/ofString body)
                                                    (instance? File body)
                                                    (HttpRequest$BodyPublishers/ofFile
                                                     (.toPath ^File body))
                                                    (instance? InputStream body)
                                                    (HttpRequest$BodyPublishers/ofInputStream
                                                     (reify Supplier (get [_] body)))))
                                         (cond->
                                             headers ((fn [^HttpRequest$Builder builder]
                                                        (->> (dissoc headers :content-length)
                                                             (reduce (fn [^HttpRequest$Builder builder [header value]]
                                                                       (.header builder (name header)
                                                                                (str value)))
                                                                     builder))))))
        builder (if timeout
                  (.timeout builder (Duration/ofMillis ^long timeout))
                  builder)
        request (.build builder)
        body-handler (case body-as
                       :input-stream
                       (HttpResponse$BodyHandlers/ofInputStream)
                       :string
                       (HttpResponse$BodyHandlers/ofString))
        response (p/promise)]
    (-> client
        (.sendAsync request body-handler)
        (.thenApply (reify Function
                      (apply [_ resp]
                        (p/resolve! response (parse-response resp)))))
        (.exceptionally (reify Function
                          (apply [_ e]
                            (p/reject! response e)))))
    response))

(defn- server-ssl-context
  [tls]
  (fides/validate tls :trust? false)
  (let [{:keys [cert key trust]} (fides/->input-streams tls)]
    (-> (cond-> (SslContextBuilder/forServer ^InputStream cert ^InputStream key)
          trust
          (-> (.clientAuth (ClientAuth/REQUIRE))
              (.trustManager ^InputStream trust)))
        (.sslProvider SslProvider/JDK)
        (.ciphers Http2SecurityUtil/CIPHERS SupportedCipherSuiteFilter/INSTANCE)
        (.applicationProtocolConfig
         (ApplicationProtocolConfig.
          ApplicationProtocolConfig$Protocol/ALPN
          ApplicationProtocolConfig$SelectorFailureBehavior/NO_ADVERTISE
          ApplicationProtocolConfig$SelectedListenerFailureBehavior/ACCEPT
          ^"[Ljava.lang.String;" (into-array [ApplicationProtocolNames/HTTP_2])))
        .build)))

(defn- client-ssl-context
  [tls]
  (fides/validate tls :trust? true)
  (let [{:keys [cert key trust]} (fides/->input-streams tls)
        ^JdkSslContext ssl-context
        (-> (SslContextBuilder/forClient)
            (.keyManager ^InputStream cert ^InputStream key)
            (.trustManager ^InputStream trust)
            .build)]
    (.context ssl-context)))
