;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.netty.http1.server
  (:require [vectio.netty :as n]
            [vectio.netty.websocket :as nws]
            [clojure.string :as st]
            [fluxus.promise :as p]
            [utilis.map :refer [map-keys]]
            [spectator.log :as log])
  (:import [io.netty.channel
            ChannelDuplexHandler
            ChannelPipeline
            ChannelHandler
            ChannelHandlerContext
            ChannelPromise]
           [java.util.concurrent ExecutorService]
           [java.io ByteArrayInputStream File]
           [io.netty.buffer ByteBuf]
           [io.netty.handler.codec.http
            DefaultHttpHeaders
            DefaultFullHttpResponse
            FullHttpRequest
            HttpObjectAggregator
            HttpRequestDecoder
            HttpRequestEncoder
            HttpVersion
            HttpResponseStatus
            HttpRequest
            HttpServerCodec]
           [io.netty.handler.ssl SslHandler]
           [io.netty.handler.codec.http.websocketx
            WebSocketServerHandshakerFactory]))

(defn websocket-request?
  [req]
  (boolean
   (and (= "websocket" (st/lower-case (str (get-in req [:headers "upgrade"]))))
        (= "upgrade" (st/lower-case (str (get-in req [:headers "connection"])))))))

(defn http1-request
  [^ChannelHandlerContext ctx ^FullHttpRequest msg]
  (let [headers (->> (.headers msg)
                     .entries
                     (map (fn [[k v]]
                            [(st/lower-case (str k)) (str v)]))
                     (into {}))
        path (.uri msg)
        [uri query-string] (st/split path #"\?")
        peer-cert-chain (try
                          (when-let [^SslHandler ssl-handler (some-> ctx .channel .pipeline (.get "ssl-handler"))]
                            (some-> ssl-handler .engine .getSession
                                    .getPeerCertificates seq))
                          (catch Exception e
                            (log/error [::http1-request :peer-cert :error] e)
                            nil))
        body (when-let [^ByteBuf content (.content msg)]
               (when (pos? (.readableBytes content))
                 (let [stream (ByteArrayInputStream.
                               (n/byte-buf-to-bytes content))]
                   (.release content)
                   stream)))]
    {:headers headers
     :peer-cert-chain peer-cert-chain
     :uri uri
     :path path
     :protocol-version (-> (.protocolVersion msg)
                           .text
                           st/lower-case)
     :scheme (if (websocket-request? {:headers headers})
               :wss
               :https)
     :body body
     :query-string query-string
     :request-method (-> (.method msg)
                         .name
                         st/lower-case
                         keyword)
     :query-params (when query-string
                     (->> (st/split query-string #"&")
                          (map #(st/split % #"\="))
                          (into {})))}))

(defn test-array
  [t]
  (let [check (type (t []))]
    (fn [arg] (instance? check arg))))

(def byte-array?
  (test-array byte-array))

(defn configure-websocket-handlers
  [^ChannelHandlerContext ctx ^ExecutorService exec-service initial-request
   {:keys [on-open on-close
           on-text-message
           on-binary-message
           on-ping-message
           on-pong-message]}]
  (let [ready (p/promise)
        inbound-handler (nws/inbound-handler
                         exec-service
                         {:on-text-message on-text-message
                          :on-binary-message on-binary-message
                          :on-ping-message on-ping-message
                          :on-pong-message on-pong-message
                          :on-close on-close})
        max-flush-size (dec (int (Math/pow 2 16)))
        max-frame-size (dec (int (Math/pow 2 16)))]
    (when on-close
      (n/with-promise-listener (.closeFuture (.channel ctx))
        (fn []
          (on-close))))
    (doto (.pipeline ctx)
      (n/safe-remove-handler "request-encoder")
      (n/safe-remove-handler "request-decoder")
      (n/safe-remove-handler "aggregator")
      (n/safe-remove-handler "request-handler")
      (.addLast "frame-handler" ^ChannelDuplexHandler inbound-handler))
    (fn []
      (try
        (on-open {:initial-request initial-request
                  :send #(n/safe-execute
                          ctx (fn []
                                (->> %
                                     (nws/data->websocket-frames ctx max-frame-size)
                                     (nws/send-websocket-frames ctx max-flush-size))))
                  :close (fn [] (n/safe-execute ctx (fn [] (.close (.channel ctx)))))})
        (p/resolve! ready true)
        (catch Exception e
          (log/error [::websocket-handler-config :error] e)
          (p/reject! ready e))))))

(defn handle-websocket-handshake
  ^ChannelPromise [^ChannelHandlerContext ctx ^HttpRequest req]
  (let [websocket-url (str "wss://"
                           (.get (.headers req) "Host")
                           (.uri req))
        ws-factory (WebSocketServerHandshakerFactory. websocket-url nil true)
        handshaker (.newHandshaker ws-factory req)]
    (if (not handshaker)
      (WebSocketServerHandshakerFactory/sendUnsupportedVersionResponse (.channel ctx))
      (.handshake handshaker (.channel ctx) req))))

(defn http1-request-handler
  ^ChannelHandler [^ExecutorService exec-service handler]
  (proxy [ChannelDuplexHandler] []
    (channelRead [^ChannelHandlerContext ctx msg]
      (if (instance? FullHttpRequest msg)
        (let [^HttpRequest msg msg
              req (http1-request ctx msg)]
          (or (when (websocket-request? req)
                (let [response (handler req)]
                  (when (= 101 (:status response))
                    (let [{:keys [handlers initial-request]} (meta response)
                          on-open (configure-websocket-handlers
                                   ctx exec-service
                                   initial-request handlers)
                          ^ChannelPromise handshake-future (handle-websocket-handshake ctx msg)]
                      (.submit exec-service
                               (reify Runnable
                                 (run [_]
                                   (try (.sync handshake-future)
                                        (on-open)
                                        (catch Exception e
                                          (log/error [::http1-request-handler :run :error] e)))))))
                    true)))
              (n/run exec-service
                (fn []
                  (try (let [response (try (handler req)
                                           (catch Exception e
                                             (log/error [::http1-request-handler] e)
                                             {:status 500}))
                             response (or response
                                          {:status 404})
                             response (cond
                                        (and (not (get-in response [:headers "Content-Length"]))
                                             (not (get-in response [:headers "content-length"])))
                                        (let [body (:body response)]
                                          (cond

                                            (not body)
                                            (assoc-in response [:headers "Content-Length"]
                                                      0)

                                            (instance? File body)
                                            (assoc-in response [:headers "Content-Length"]
                                                      (.length ^File body))

                                            (or (bytes? body) (string? body))
                                            (assoc-in response [:headers "Content-Length"]
                                                      (count body))

                                            :else response))

                                        (not (:status response))
                                        (assoc :status 200)

                                        :else response)
                             netty-response (DefaultFullHttpResponse.
                                             HttpVersion/HTTP_1_1
                                             (HttpResponseStatus/valueOf (:status response))
                                             (n/to-byte-buf (:body response))
                                             (let [headers (DefaultHttpHeaders.)]
                                               (doseq [[^String k v] (map-keys #(st/lower-case
                                                                                 (str (if (keyword? %)
                                                                                        (name %)
                                                                                        %)))
                                                                               (:headers response))]
                                                 (.add headers k (str v)))
                                               headers)
                                             (DefaultHttpHeaders.))]
                         (n/safe-execute ctx #(.writeAndFlush ctx netty-response)))
                       (catch Exception e
                         (log/error [::http1-request-handler :error] e)))))))
        (let [^ChannelDuplexHandler this this]
          (proxy-super channelRead ctx msg))))))

(defn configure-http1-request-handler
  [^ChannelPipeline pipeline ^ExecutorService exec-service handler]
  (doto pipeline
    (.addLast "request-encoder" (HttpRequestEncoder.))
    (.addLast "request-decoder" (HttpRequestDecoder.))
    (.addLast "aggregator" (HttpObjectAggregator.
                            ;; 2GB
                            (Integer/MAX_VALUE)))
    (.addLast "request-handler" (http1-request-handler exec-service handler))))

(defn configure-http1-pipeline
  [^ChannelPipeline pipeline ^ExecutorService exec-service handler]
  (doto pipeline
    (.addLast "frame-codec" (HttpServerCodec.))
    (configure-http1-request-handler exec-service handler)))
