;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.util.stream
  (:require [fluxus.flow :as f]
            [fluxus.promise :as p]
            [utilis.timer :as timer]
            #?(:cljs [utilis.js :as j])))

(defn reconnecting-stream
  ([init-fn]
   (reconnecting-stream
    init-fn {:reconnect-timeout
             (fn [attempt]
               (min 5000 (* attempt 100)))}))
  ([init-fn {:keys [reconnect-timeout]}]
   (let [event-stream (f/flow)
         [client-stream internal] (f/entangled)
         underlying-stream-atom (atom nil)
         reconnect? (atom true)
         initial (atom true)
         last-state (atom :init)
         cleanup (fn []
                   (reset! reconnect? false)
                   (f/close! event-stream)
                   (when-let [stream @underlying-stream-atom]
                     (f/close! stream)))
         handle-disconnect (fn [reason]
                             (when (not= :disconnected @last-state)
                               (f/put! event-stream
                                       {:state :disconnected
                                        :error reason})
                               (reset! last-state :disconnected)))
         init (fn init
                ([] (init 0))
                ([attempt]
                 (-> (init-fn)
                     (p/then
                      (fn [underlying-stream]
                        (reset! underlying-stream-atom underlying-stream)
                        (f/put! event-stream
                                {:underlying-stream underlying-stream
                                 :stream client-stream
                                 :initial @initial
                                 :state :connected})
                        (reset! last-state :connected)
                        (f/consume (fn [message] (f/put! internal message)) underlying-stream)
                        (f/on-close underlying-stream
                                    (fn [_]
                                      (reset! underlying-stream-atom nil)
                                      (handle-disconnect :underlying-stream-closed)
                                      (when @reconnect? (init))))
                        (when @initial (reset! initial false))))
                     (p/catch
                         (fn [error]
                           (handle-disconnect error)
                           (when @reconnect?
                             (timer/run-after
                              #(init (inc attempt))
                              (cond
                                (number? reconnect-timeout)
                                reconnect-timeout

                                (fn? reconnect-timeout)
                                (reconnect-timeout attempt)

                                :else (throw (ex-info "reconnect-timeout must be a number or a function"
                                                      {:reconnect-timeout reconnect-timeout}))))))))))]
     (f/consume #(when-let [stream @underlying-stream-atom]
                   (f/put! stream %))
                internal)
     (f/on-close internal (fn [_] (cleanup)))
     (f/on-close event-stream (fn [_] (cleanup)))
     (init)
     event-stream)))
