;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.http
  (:refer-clojure :exclude [get])
  (:require [vectio.netty.server :as server]
            [vectio.netty.http1.websocket :as http1-websocket]
            [hato.client :as hc]
            [utilis.fn :refer [fsafe]]
            [clojure.java.io :as io]
            [integrant.core :as ig])
  (:import [java.net InetSocketAddress]
           [javax.net.ssl SSLParameters]
           [io.netty.handler.ssl
            ApplicationProtocolConfig
            ApplicationProtocolConfig$Protocol
            ApplicationProtocolConfig$SelectorFailureBehavior
            ApplicationProtocolConfig$SelectedListenerFailureBehavior
            ApplicationProtocolNames
            SslContextBuilder
            SslProvider
            ClientAuth
            SupportedCipherSuiteFilter
            JdkSslContext]
           [io.netty.handler.codec.http2 Http2SecurityUtil]
           [java.net InetSocketAddress]
           [java.io InputStream ByteArrayInputStream File Closeable]))

(declare server ensure-http-client server-ssl-context client-ssl-context)

(defmethod ig/init-key :vectio.http/server
  [_ {:keys [host port ring-handler] :as opts}]
  (when (not port)
    (throw (ex-info "A port must be provided to start :vectio.http/server"
                    {:host host :port port})))
  {:ring-handler ring-handler
   :http-server (server opts)})

(defmethod ig/halt-key! :vectio.http/server
  [_ {:keys [http-server]}]
  (when http-server
    (.close ^Closeable http-server)))

(defmethod ig/suspend-key! :vectio.http/server
  [_ {:keys [ring-handler]}]
  (reset! ring-handler (promise)))

(defmethod ig/resume-key :vectio.http/server
  [key opts old-opts old-impl]
  (if (= (dissoc opts :ring-handler) (dissoc old-opts :ring-handler))
    (do (deliver @(:ring-handler old-impl) (:ring-handler opts))
        old-impl)
    (do (ig/halt-key! key old-impl)
        (ig/init-key key opts))))

(defn server
  [{:keys [host port
           ring-handler
           tls ssl-context]
    :as server-opts}]
  (server/start-server
   (merge
    (select-keys server-opts [:leak-detector-level
                              :initial-window-size
                              :websocket-max-frame-size
                              :max-frame-size
                              :max-flush-size
                              :max-concurrent-streams
                              :max-header-list-size
                              :push-enabled
                              :default-outbound-max-frame-size
                              :protocols])
    {:handler ring-handler
     :socket-address (InetSocketAddress. ^String host ^int port)
     :ssl-context (or ssl-context
                      (when (not-empty tls)
                        (server-ssl-context tls)))})))

(defn websocket-client
  [{:keys [host port tls path]
    :as websocket-args}]
  (http1-websocket/websocket-client-stream websocket-args))

(defn websocket-stream-response
  [request]
  (server/websocket-stream-response request))

(defn client
  [{:keys [tls ssl-context]}]
  (hc/build-http-client
   (merge
    {:connect-timeout 10000}
    (when (or tls ssl-context)
      {:ssl-parameters
       (doto (SSLParameters.)
         (.setNeedClientAuth true))
       :ssl-context
       (or ssl-context
           (when (not-empty tls)
             (client-ssl-context tls)))}))))

(defn get
  ([client url] (get client url nil))
  ([client url options]
   (hc/get url (assoc options :http-client client))))

(defn post
  ([client url] (post client url nil))
  ([client url options]
   (hc/post url (assoc options :http-client client))))

(defn put
  ([client url] (put client url nil))
  ([client url options]
   (hc/put url (assoc options :http-client client))))

(defn patch
  ([client url] (patch client url nil))
  ([client url options]
   (hc/patch url (assoc options :http-client client))))

(defn delete
  ([client url] (delete client url nil))
  ([client url options]
   (hc/delete url (assoc options :http-client client))))

(defn head
  ([client url] (head client url nil))
  ([client url options]
   (hc/head url (assoc options :http-client client))))

(defn options
  ([client url] (options client url nil))
  ([client url options]
   (hc/options url (assoc options :http-client client))))


;;; Private

(defn- ->input-stream ^InputStream
  [v]
  (cond
    (instance? InputStream v) v
    (string? v) (ByteArrayInputStream. (.getBytes ^String v))
    (instance? File v) (io/input-stream v)))

(defn- server-ssl-context
  [tls]
  (-> (cond-> (SslContextBuilder/forServer
               ^InputStream (->input-stream (get-in tls [:local :cert]))
               ^InputStream (->input-stream (get-in tls [:local :key])))
        (get-in tls [:remote :trust])
        (-> (.clientAuth (ClientAuth/REQUIRE))
            (.trustManager ^InputStream (->input-stream (get-in tls [:remote :trust])))))
      (.sslProvider SslProvider/JDK)
      (.ciphers Http2SecurityUtil/CIPHERS SupportedCipherSuiteFilter/INSTANCE)
      (.applicationProtocolConfig
       (ApplicationProtocolConfig.
        ApplicationProtocolConfig$Protocol/ALPN
        ApplicationProtocolConfig$SelectorFailureBehavior/NO_ADVERTISE
        ApplicationProtocolConfig$SelectedListenerFailureBehavior/ACCEPT
        ^"[Ljava.lang.String;" (into-array [ApplicationProtocolNames/HTTP_2])))
      .build))

(defn- client-ssl-context
  [tls]
  (let [^JdkSslContext ssl-context
        (-> (SslContextBuilder/forClient)
            (.trustManager (->input-stream (get-in tls [:remote :trust])))
            (.keyManager (->input-stream (get-in tls [:local :cert]))
                         (->input-stream (get-in tls [:local :key])))
            .build)]
    (.context ssl-context)))
