;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.netty.h2.handlers.headers
  (:require [vectio.netty.h2.handlers.websocket :as ws]
            [vectio.netty :as n]
            [utilis.map :refer [map-keys assoc-if]]
            [utilis.types.number :refer [string->long]]
            [clojure.string :as st]
            [fluxus.promise :as p])
  (:import [java.util Map$Entry]
           [java.util.concurrent ExecutorService]
           [java.io PipedOutputStream PipedInputStream File]
           [io.netty.buffer ByteBuf]
           [io.netty.handler.codec.http QueryStringDecoder]
           [io.netty.channel
            ChannelHandlerContext
            ChannelPipeline
            ChannelPromise]
           [io.netty.handler.stream ChunkedFile]
           [io.netty.handler.codec.http2
            Http2Error
            Http2HeadersFrame
            Http2Headers
            Http2FrameStream
            DefaultHttp2DataFrame
            DefaultHttp2Headers
            DefaultHttp2HeadersFrame
            DefaultHttp2ResetFrame
            Http2DataFrame]))

(defn add-all-headers
  [^Http2Headers headers headers-map]
  (doseq [[^String k v] (map-keys #(st/lower-case
                                    (str (if (keyword? %)
                                           (name %)
                                           %)))
                                  headers-map)]
    (.add headers k (str v)))
  headers)

(defn headers-frame->headers
  [^Http2HeadersFrame msg]
  (->> (.headers msg)
       .iterator
       iterator-seq
       (reduce (fn [m ^Map$Entry entry]
                 (assoc! m
                         (.toString (.getKey entry))
                         (.toString (.getValue entry))))
               (transient {}))
       persistent!))

(defn request-method
  [headers]
  (case (get headers ":method")
    "GET" :get
    "HEAD" :head
    "POST" :post
    "PUT" :put
    "DELETE" :delete
    "CONNECT" :connect
    "OPTIONS" :options
    "TRACE" :trace
    "PATCH" :patch))

(defn scheme
  [headers]
  (case (get headers ":scheme")
    "http" :http
    "https" :https
    "ws" :ws
    "wss" :wss))

(defn parse-query-string
  [^String query-string]
  (->> (st/split query-string #"&")
       (remove empty?)
       (reduce (fn [query-params kv-string]
                 (let [[k v] (st/split kv-string #"\=")]
                   (assoc! query-params k v)))
               (transient {}))
       persistent!
       not-empty))

(defn uri
  [headers]
  (-> headers
      (get ":path")
      (st/split #"\?")
      first))

(defn create-body-input-stream
  [request]
  (let [content-length (get-in request [:headers "content-length"])
        content-length (when (string? content-length)
                         (string->long content-length))]
    (when (not content-length)
      (throw (ex-info "No content-length header provided in upload"
                      {:request request})))
    (if (pos? content-length)
      (let [in (PipedInputStream. (int content-length))]
        (assoc request
               :body in
               :body-output-stream (PipedOutputStream. in)))
      request)))

(defn headers-frame->request
  [^Http2HeadersFrame msg]
  (let [headers (headers-frame->headers msg)
        _ (when (or (not (get headers ":path"))
                    (not (get headers ":scheme"))
                    (not (get headers ":authority")))
            (throw (ex-info "An http2 headers frame must include a :path, :scheme and :authority."
                            {:headers headers})))
        ^String path (get headers ":path")
        request-method (request-method headers)
        query-string (.rawQuery (QueryStringDecoder. path))]
    (cond-> {:headers headers
             :protocol-version "http/2"
             :uri (uri headers)
             :path path
             :scheme (scheme headers)
             :query-string query-string
             :request-method request-method
             :query-params (when query-string (parse-query-string query-string))}
      (#{:put :post} request-method) (create-body-input-stream))))

(defn handle?
  [msg]
  (instance? Http2HeadersFrame msg))

(defn response->headers
  [{:keys [status headers]}]
  (doto (DefaultHttp2Headers.)
    (.status (str status))
    (add-all-headers headers)))

(defn response->response-headers-frame
  ^Http2HeadersFrame [_connection-state ^Http2FrameStream stream request {:keys [body] :as response} response-data-frame-count]
  (-> (response->headers response)
      (DefaultHttp2HeadersFrame.
       (boolean
        (and (not (ws/websocket-request? request))
             (or (zero? response-data-frame-count)
                 (nil? body)
                 (and (instance? File body)
                      (zero? (.length ^File body)))))))
      (.stream stream)))

(defn file->data-frames
  [^ChannelHandlerContext ctx ^Http2FrameStream stream ^File file ^long max-frame-size]
  (when (pos? (.length file))
    (let [chunked-file (ChunkedFile. ^File file max-frame-size)
          allocator (.alloc ctx)
          next-frame (fn next-frame []
                       (if (not (.isEndOfInput chunked-file))
                         (lazy-seq
                          (cons (-> (.readChunk chunked-file allocator)
                                    (DefaultHttp2DataFrame.
                                     (boolean (.isEndOfInput chunked-file)))
                                    (.stream stream))
                                (next-frame)))
                         (do (.close chunked-file)
                             nil)))]
      (next-frame))))

(defn response->response-data-frames
  [connection-state ^ChannelHandlerContext ctx stream _request {:keys [body]}]
  (when (not (nil? body))
    (let [max-frame-size (get-in @connection-state [:settings :settings-max-frame-size])]
      (if (instance? File body)
        (file->data-frames ctx stream ^File body max-frame-size)
        (let [^ByteBuf byte-buf (n/to-byte-buf ctx body)
              frames (->> max-frame-size
                          (n/byte-buf-to-http2-data-frames stream byte-buf)
                          (mapv n/acquire))]
          (n/release byte-buf)
          frames)))))

(defn setup-stream-window
  [connection-state ^ChannelPipeline pipeline ^Http2HeadersFrame msg stream-id request]
  (let [window-size (get-in @connection-state [:settings :settings-initial-window-size])
        backlog (volatile! [])
        closed? (atom false)
        send-close-frame (fn []
                           (try
                             (->> (-> Http2Error/STREAM_CLOSED
                                      (DefaultHttp2ResetFrame.)
                                      (.stream (.stream msg)))
                                  (n/invoke-write pipeline "frame-writer"))
                             (catch Exception e
                               (println "Exception occurred in close-handler" e))))
        cleanup-handlers (atom [(fn []
                                  (doseq [[_ ^Http2DataFrame msg _] @backlog]
                                    (try (n/release msg)
                                         (catch Exception e
                                           (println "Exception releasing message from backlog" e))))
                                  (vreset! backlog nil))])
        handle-close (fn []
                       (let [close? (locking closed?
                                      (if @closed?
                                        false
                                        (do (reset! closed? true)
                                            true)))]
                         (when close?
                           (doseq [handler @cleanup-handlers]
                             (try
                               (handler)
                               (catch Exception e
                                 (println "Exception occurred in cleanup handler" e))))
                           (swap! connection-state update :streams dissoc stream-id))))]
    (n/with-promise-listener (.closeFuture (.channel pipeline))
      (fn []
        (handle-close)))
    (swap! connection-state update-in
           [:streams stream-id]
           (fn [stream]
             (assoc stream
                    :request request
                    :window (volatile! window-size)
                    :backlog backlog
                    :cleanup-handlers cleanup-handlers
                    :close (fn [& {:keys [send-close]
                                  :or {send-close true}}]
                             (when send-close
                               (send-close-frame))
                             (handle-close)))))))

(defn setup-websocket-handler
  [^ChannelHandlerContext ctx connection-state ^Http2HeadersFrame msg stream-id response]
  (when-let [data-frame-handler (ws/init-handler ctx connection-state msg response)]
    (swap! connection-state assoc-in
           [:streams stream-id :data-frame-handler]
           data-frame-handler)))

(defn setup-upload-handler
  [^ChannelHandlerContext _ctx connection-state ^Http2HeadersFrame _msg stream-id request]
  (let [{:keys [body-output-stream]} request
        ^PipedOutputStream body-output-stream body-output-stream
        close-bos (fn []
                    (try (.close body-output-stream)
                         (catch Exception e
                           (println "Exception occurred closing body output stream" e))))
        done-p (p/promise)
        upload-total (atom 0)]
    (-> @connection-state
        (get-in [:streams stream-id :cleanup-handlers])
        (swap! (fn [handlers]
                 (conj (vec handlers)
                       (fn []
                         (close-bos)
                         (when-let [is (:body request)]
                           (try (.close ^java.io.InputStream is)
                                (catch Exception _e)))
                         (try (when (not (p/realized? done-p))
                                (p/reject! done-p :connection-closed))
                              (catch Exception e
                                (println "Exception occurred in cleanup handler" e))))))))
    (swap! connection-state assoc-in
           [:streams stream-id :data-frame-handler]
           (fn [^Http2DataFrame frame]
             (let [^ByteBuf buf (.content frame)
                   bytes (byte-array (.readableBytes buf))]
               (.readBytes buf bytes)
               (swap! upload-total + (count bytes))
               ;; possible blocking operation? Move to another writer thread pool?
               (.write body-output-stream bytes 0 (count bytes))
               (.flush body-output-stream)
               (when (.isEndStream frame)
                 (close-bos)
                 (p/resolve! done-p :done)))))
    done-p))

(defn handle
  [^ChannelHandlerContext ctx connection-state ^Http2HeadersFrame msg]
  (let [{:keys [handler exec-service peer-cert-chain]} @connection-state
        ^ExecutorService exec-service exec-service
        ^ChannelPipeline pipeline (.pipeline ctx)
        stream (.stream msg)
        stream-id (.id stream)
        upload? (boolean (#{"PUT" "POST"} (str (.get (.headers msg) ":method"))))
        completed (p/promise)
        close-stream (fn [send-close]
                       (when-let [close (get-in @connection-state [:streams stream-id :close])]
                         (n/safe-execute
                          ctx (fn []
                                (close :send-close send-close)))))
        ;; guard against stream reuse
        _ (when (get-in @connection-state [:streams stream-id])
            (close-stream false))
        write-response (fn [response-headers-frame response-data-frames]
                         (n/safe-execute
                          ctx
                          (fn []
                            (let [byte-counter (volatile! 0)
                                  close-stream (fn [^ChannelPromise p]
                                                 (if p
                                                   (n/with-promise-listener p
                                                     (fn []
                                                       (close-stream true)))
                                                   (close-stream false)))]
                              (let [^Http2HeadersFrame frame response-headers-frame
                                    ^ChannelPromise p (n/invoke-write pipeline "frame-writer" frame)]
                                (when (.isEndStream frame)
                                  (close-stream p)))
                              (doseq [^Http2DataFrame frame response-data-frames]
                                (let [^ChannelPromise p (n/invoke-write pipeline "frame-writer" frame)]
                                  (vswap! byte-counter + (.initialFlowControlledBytes ^Http2DataFrame frame))
                                  (when (> @byte-counter 1E5)
                                    (vreset! byte-counter 0)
                                    (.flush ctx))
                                  (when (.isEndStream frame)
                                    (close-stream p)))))
                            (.flush ctx)
                            (p/resolve! completed true))))
        send-500 (fn []
                   (write-response
                    (response->response-headers-frame
                     connection-state
                     stream
                     nil
                     {:status 500
                      :body "An internal error occurred."}
                     0)
                    nil))]
    (try (let [request (assoc-if (headers-frame->request msg) :peer-cert-chain peer-cert-chain)
               window? (boolean
                        (or (ws/websocket-request? request)
                            (#{:put :post} (:request-method request))))
               _ (when window? (setup-stream-window connection-state pipeline msg stream-id request))
               done-upload (when upload? (setup-upload-handler ctx connection-state msg stream-id request))]
           (n/run exec-service
             (fn []
               (try (let [response (assoc (handler request) :completed completed)
                          response-data-frames (response->response-data-frames connection-state ctx stream request response)
                          response-headers-frame (response->response-headers-frame connection-state stream request response (count response-data-frames))
                          response-byte-count (reduce (fn [c ^Http2DataFrame frame]
                                                        (+ c (.initialFlowControlledBytes frame)))
                                                      0
                                                      response-data-frames)
                          window? (boolean
                                   (or window?
                                       (> response-byte-count
                                          (get-in @connection-state [:settings :settings-initial-window-size]))))
                          write-response (fn [] (write-response response-headers-frame response-data-frames))]
                      (when (and window? (not (get-in @connection-state [:streams stream-id :window])))
                        (setup-stream-window connection-state pipeline msg stream-id request))
                      (when (ws/websocket-request? request)
                        (setup-websocket-handler ctx connection-state msg stream-id response))
                      (if done-upload
                        (-> done-upload
                            (p/then (fn [_]
                                      (write-response)))
                            (p/catch (fn [error]
                                       (locking Object
                                         (println "Error occurred in h2 upload handler" error)))))
                        (write-response)))
                    (catch Exception e
                      (try (send-500)
                           (.fireExceptionCaught ctx e)
                           (catch Exception _)))))))
         (catch Exception e
           (try (send-500)
                (.fireExceptionCaught ctx e)
                (catch Exception _)))))
  nil)
