;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.http
  (:require [vectio.netty.server :as server]
            [vectio.netty.http1.websocket :as http1-websocket]
            [vectio.netty.h2.client :as h2-client]
            [clojure.set :refer [rename-keys]]
            [utilis.fn :refer [fsafe]]
            [clojure.java.io :as io]
            [integrant.core :as ig])
  (:import [java.net InetSocketAddress]
           [io.netty.handler.ssl
            ApplicationProtocolConfig
            ApplicationProtocolConfig$Protocol
            ApplicationProtocolConfig$SelectorFailureBehavior
            ApplicationProtocolConfig$SelectedListenerFailureBehavior
            ApplicationProtocolNames
            SslContextBuilder
            SslProvider
            ClientAuth
            SupportedCipherSuiteFilter]
           [io.netty.handler.codec.http2 Http2SecurityUtil]
           [java.net InetSocketAddress]
           [java.io Closeable]))

(defn server
  [{:keys [ssl-context
           socket-address
           host port
           handler
           leak-detector-level]
    :as server-args}]
  (server/start-server
   (cond-> server-args
     (and (not socket-address) host port)
     (update :socket-address
             #(or % (InetSocketAddress. ^String host (int port)))))))

(defn client
  []
  (h2-client/http-client))

(defn websocket-client
  [{:keys [host
           port
           ssl-context
           path]
    :as websocket-args}]
  (http1-websocket/websocket-client-stream websocket-args))

(defn websocket-stream-response
  [request]
  (server/websocket-stream-response request))


;;; Integrant

(defmethod ig/init-key :vectio.http/server
  [_ opts]
  (let [opts (rename-keys opts {:port :http-port
                                :host :http-host})
        ring-handler (atom (delay (:ring-handler opts)))
        {:keys [http-host http-port tls] :or {http-host "localhost"}} opts]
    (when (not http-port)
      (throw (ex-info "An http-port must be provided to start :vectio.http/server"
                      {:http-host http-host
                       :http-port http-port})))
    {:ring-handler ring-handler
     :http-server (server
                   (merge
                    (select-keys opts [:leak-detector-level
                                       :initial-window-size
                                       :max-frame-size
                                       :max-flush-size
                                       :max-concurrent-streams
                                       :max-header-list-size
                                       :push-enabled
                                       :default-outbound-max-frame-size])
                    {:handler (fn [request] ((fsafe @@ring-handler) request))
                     :socket-address (InetSocketAddress. ^String http-host ^int http-port)
                     :ssl-context (-> (cond-> (SslContextBuilder/forServer
                                               (io/file (get-in tls [:server :cert]))
                                               (io/file (get-in tls [:server :key])))
                                        (get-in tls [:client :ca])
                                        (-> (.clientAuth (ClientAuth/REQUIRE))
                                            (.trustManager (io/file (get-in tls [:client :ca])))))
                                      (.sslProvider SslProvider/JDK)
                                      (.ciphers Http2SecurityUtil/CIPHERS SupportedCipherSuiteFilter/INSTANCE)
                                      (.applicationProtocolConfig
                                       (ApplicationProtocolConfig.
                                        ApplicationProtocolConfig$Protocol/ALPN
                                        ApplicationProtocolConfig$SelectorFailureBehavior/NO_ADVERTISE
                                        ApplicationProtocolConfig$SelectedListenerFailureBehavior/ACCEPT
                                        ^"[Ljava.lang.String;" (into-array [ApplicationProtocolNames/HTTP_2])))
                                      .build)}))}))

(defmethod ig/halt-key! :vectio.http/server
  [_ {:keys [http-server]}]
  (when http-server
    (.close ^Closeable http-server)))

(defmethod ig/suspend-key! :vectio.http/server
  [_ {:keys [ring-handler]}]
  (reset! ring-handler (promise)))

(defmethod ig/resume-key :vectio.http/server
  [key opts old-opts old-impl]
  (if (= (dissoc opts :ring-handler) (dissoc old-opts :ring-handler))
    (do (deliver @(:ring-handler old-impl) (:ring-handler opts))
        old-impl)
    (do (ig/halt-key! key old-impl)
        (ig/init-key key opts))))
