;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.netty.http1.server
  (:require [vectio.netty :as n]
            [vectio.netty.websocket :as nws]
            [clojure.string :as st]
            [com.brunobonacci.mulog :as u]
            [manifold.deferred :as d]
            [utilis.map :refer [map-keys]])
  (:import [io.netty.channel
            ChannelDuplexHandler
            ChannelPipeline
            ChannelHandler
            ChannelHandlerContext
            ChannelPromise]
           [java.util.concurrent ExecutorService]
           [io.netty.handler.codec.http
            DefaultHttpHeaders
            DefaultFullHttpResponse
            HttpVersion
            HttpResponseStatus
            HttpRequest
            HttpServerCodec]
           [io.netty.handler.ssl SslHandler]
           [io.netty.handler.codec.http.websocketx
            WebSocketServerHandshakerFactory]))

(defn websocket-request?
  [req]
  (boolean
   (and (= "websocket" (st/lower-case (str (get-in req [:headers "upgrade"]))))
        (= "upgrade" (st/lower-case (str (get-in req [:headers "connection"])))))))

(defn http1-request
  [^ChannelHandlerContext ctx ^HttpRequest msg]
  (let [headers (->> (.headers msg)
                     .entries
                     (map (fn [[k v]]
                            [(st/lower-case (str k)) (str v)]))
                     (into {}))
        path (.uri msg)
        [uri query-string] (st/split path #"\?")
        peer-cert-chain (try
                          (when-let [^SslHandler ssl-handler (some-> ctx .channel .pipeline (.get "ssl-handler"))]
                            (some-> ssl-handler .engine .getSession
                                    .getPeerCertificates seq))
                          (catch Exception _ nil))]
    {:headers headers
     :peer-cert-chain peer-cert-chain
     :uri uri
     :path path
     :protocol-version (-> (.protocolVersion msg)
                           .text
                           st/lower-case)
     :scheme (if (websocket-request? {:headers headers})
               :wss
               :https)
     :query-string query-string
     :request-method (-> (.method msg)
                         .name
                         st/lower-case
                         keyword)
     :query-params (when query-string
                     (->> (st/split query-string #"&")
                          (map #(st/split % #"\="))
                          (into {})))}))

(defn test-array
  [t]
  (let [check (type (t []))]
    (fn [arg] (instance? check arg))))

(def byte-array?
  (test-array byte-array))

(defn configure-websocket-handlers
  [^ChannelHandlerContext ctx ^ExecutorService exec-service initial-request
   {:keys [on-open on-close
           on-text-message
           on-binary-message
           on-ping-message
           on-pong-message]}]
  (let [ready (d/deferred)
        inbound-handler (nws/inbound-handler
                         exec-service
                         {:on-text-message on-text-message
                          :on-binary-message on-binary-message
                          :on-ping-message on-ping-message
                          :on-pong-message on-pong-message
                          :on-close on-close})
        max-flush-size (dec (int (Math/pow 2 16)))
        max-frame-size (dec (int (Math/pow 2 16)))]
    (when on-close
      (n/with-promise-listener (.closeFuture (.channel ctx))
        (fn []
          (on-close))))
    (doto (.pipeline ctx)
      (.remove "request-handler")
      (.addLast "frame-handler" ^ChannelDuplexHandler inbound-handler))
    (fn []
      (try
        (on-open {:initial-request initial-request
                  :send #(n/safe-execute
                          ctx (fn []
                                (->> %
                                     (nws/data->websocket-frames ctx max-frame-size)
                                     (nws/send-websocket-frames ctx max-flush-size))))
                  :close (fn [] (n/safe-execute ctx (fn [] (.close (.channel ctx)))))})
        (d/success! ready true)
        (catch Exception e
          (u/log ::websocket-handler-config :exception e)
          (d/error! ready e))))))

(defn handle-websocket-handshake
  ^ChannelPromise [^ChannelHandlerContext ctx ^HttpRequest req]
  (let [websocket-url (str "wss://"
                           (.get (.headers req) "Host")
                           (.uri req))
        ws-factory (WebSocketServerHandshakerFactory. websocket-url nil true)
        handshaker (.newHandshaker ws-factory req)]
    (if (not handshaker)
      (WebSocketServerHandshakerFactory/sendUnsupportedVersionResponse (.channel ctx))
      (.handshake handshaker (.channel ctx) req))))

(defn http1-request-handler
  ^ChannelHandler [^ExecutorService exec-service handler]
  (proxy [ChannelDuplexHandler] []
    (channelRead [^ChannelHandlerContext ctx msg]
      (if (instance? HttpRequest msg)
        (let [^HttpRequest msg msg
              req (http1-request ctx msg)]
          (or (when (websocket-request? req)
                (let [response (handler req)]
                  (when (= 101 (:status response))
                    (let [{:keys [handlers initial-request]} (meta response)
                          on-open (configure-websocket-handlers ctx exec-service initial-request handlers)
                          ^ChannelPromise handshake-future (handle-websocket-handshake ctx msg)]
                      (.submit exec-service
                               (reify Runnable
                                 (run [_]
                                   (try (.sync handshake-future)
                                        (on-open)
                                        (catch Exception e
                                          (locking Object
                                            (println e))))))))
                    true)))
              (n/run exec-service
                (fn []
                  (try (let [response (handler req)
                             netty-response (DefaultFullHttpResponse.
                                             HttpVersion/HTTP_1_1
                                             (HttpResponseStatus/valueOf (:status response))
                                             (n/to-byte-buf (:body response))
                                             (let [headers (DefaultHttpHeaders.)]
                                               (doseq [[^String k v] (map-keys #(st/lower-case
                                                                                 (str (if (keyword? %)
                                                                                        (name %)
                                                                                        %)))
                                                                               (:headers response))]
                                                 (.add headers k (str v)))
                                               headers)
                                             (DefaultHttpHeaders.))]
                         (n/safe-execute ctx #(.writeAndFlush ctx netty-response)))
                       (catch Exception e
                         (locking Object
                           (println e))))))))
        (let [^ChannelDuplexHandler this this]
          (proxy-super channelRead ctx msg))))))

(defn configure-http1-request-handler
  [^ChannelPipeline pipeline ^ExecutorService exec-service handler]
  (doto pipeline
    (.addLast "request-handler" (http1-request-handler exec-service handler))))

(defn configure-http1-pipeline
  [^ChannelPipeline pipeline ^ExecutorService exec-service handler]
  (doto pipeline
    (.addLast "frame-codec" (HttpServerCodec.))
    (configure-http1-request-handler exec-service handler)))
