;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.haslett.websocket
  (:refer-clojure :exclude [subs])
  (:require [vectio.util.manifold.stream :as s]
            [vectio.util.manifold.deferred :as d]
            [cljs.core.async :as async]
            [haslett.client :as ws]
            [utilis.js :as j]))

(defn websocket-client
  [{:keys [address on-text-message on-close]}]
  (let [deferred (d/deferred)]
    (async/go
      (try
        (let [return (ws/connect address)
              stream (async/<! return)]
          (if (ws/connected? stream)
            (let [{:keys [close-status source]} stream]
              (async/go
                (try (loop []
                       (async/alt!
                         source ([message]
                                 (do (on-text-message message)
                                     (recur)))
                         close-status ([status]
                                       (on-close status))))
                     (catch js/Error e
                       (js/console.error "Error occurred in vectio.haslett.websocket/websocket-client read loop" e))))
              (d/success! deferred
                          {:stream stream
                           :close (fn [] (ws/close stream))
                           :send (fn [^js/String message]
                                   (async/put! (:sink stream) message))}))
            (d/error! deferred nil)))
        (catch js/Error e
          (js/console.error "Error occurred in vectio.haslett.websocket/websocket-client" e))))
    deferred))

(defn websocket-client-stream
  [{:keys [address] :as args}]
  (let [out (s/stream)
        in (s/stream)
        spliced (s/splice out in)
        deferred (d/deferred)]
    (d/on-realized
     (websocket-client
      (assoc args
             :on-close #(s/close! spliced)
             :on-text-message #(s/put! in %)))
     (fn [{:keys [send close]}]
       (s/consume send out)
       (s/on-closed out (fn [] (close)))
       (d/success! deferred spliced))
     (fn [error]
       (s/close! spliced)
       (s/close! in)
       (d/error! deferred error)))
    deferred))
