;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.util.manifold.stream
  (:require [cljs.core.async :as async]))

(declare wrap unwrap)

(defn stream
  []
  {:chan (async/chan)
   :handlers {:close (atom [])}
   :closed (atom false)})

(defn consume
  [callback source]
  (async/go
    (loop []
      (when-let [x (async/<! (or (:source source)
                                 (:chan source)))]
        (callback (unwrap x))
        (recur)))))

(defn on-closed
  [sink callback]
  (swap! (-> sink :handlers :close) conj callback))

(defn put!
  [sink x]
  (async/put! (or (:sink sink)
                  (:chan sink))
              (wrap x)))

(defn closed?
  [sink]
  (boolean @(:closed sink)))

(defn close!
  [sink]
  (when (not @(:closed sink))
    (async/close! (:chan sink))
    (reset! (:closed sink) true)
    (doseq [handler @(-> sink :handlers :close)]
      (handler))))

(defn splice
  [sink source]
  (let [spliced (stream)]
    (on-closed spliced #(close! sink))
    (on-closed source #(close! spliced))
    (assoc spliced
           :source (:chan source)
           :sink (:chan sink))))

(defn connect
  [source sink]
  (on-closed source #(close! sink))
  (consume #(put! sink %) source))

(defn stream?
  [x]
  (and (map? x)
       (:chan x)
       (:handlers x)
       (:closed x)))


;;; Private

(defn- wrap
  [x]
  [x])

(defn- unwrap
  [x]
  (first x))
