;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.util.manifold.deferred
  (:refer-clojure :exclude [realized?])
  (:require [utilis.js :as j]))

(declare throw-on-realized on-realized error!)

(defn deferred
  []
  (let [value (atom nil)
        deferred #js {:handlers {:success (atom [])
                                 :error (atom [])}
                      :value value}]
    (js/Object.assign
     deferred
     #js {:then (fn [callback]
                  (try (on-realized deferred callback)
                       (catch :default e
                         (error! deferred e))))
          :catch (fn [callback] (on-realized deferred nil callback))
          :finally (fn [callback]
                     (on-realized deferred
                                  callback
                                  callback))})))

(defn deferred?
  [x]
  (and (map? x)
       (j/get x :handlers)
       (j/get x :value)))

(defn realized?
  [deferred]
  (boolean @(j/get deferred :value)))

(defn success!
  [deferred x]
  (let [value (j/get deferred :value)
        handlers (j/get deferred :handlers)]
    (throw-on-realized deferred)
    (reset! value {:state :success :value x})
    (doseq [handler @(:success handlers)]
      (handler x))
    deferred))

(defn error!
  [deferred x]
  (let [value (j/get deferred :value)
        handlers (j/get deferred :handlers)]
    (throw-on-realized deferred)
    (reset! value {:state :error :value x})
    (doseq [handler @(:error handlers)]
      (handler x))
    deferred))

(defn on-realized
  ([deferred on-success] (on-realized deferred on-success nil))
  ([deferred on-success on-error]
   (let [value (j/get deferred :value)
         handlers (j/get deferred :handlers)]
     (if (realized? deferred)
       (condp = (:state @value)
         :error (when on-error (on-error (:value @value)))
         :success (when on-success (on-success (:value @value))))
       (do (when on-success
             (swap! (:success handlers) conj on-success))
           (when on-error
             (swap! (:error handlers) conj on-error))))
     deferred)))


;;; Private

(defn- throw-on-realized
  [deferred]
  (when (realized? deferred)
    (throw (ex-info "Deferred has already been realized"
                    {:deferred deferred}))))
