;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns vectio.netty.h2.handlers.websocket
  (:require [vectio.netty :as n]
            [vectio.netty.websocket :as nws])
  (:import [java.util.concurrent ExecutorService]
           [io.netty.buffer ByteBuf]
           [io.netty.channel
            ChannelHandlerContext
            ChannelDuplexHandler
            ChannelPromise]
           [io.netty.handler.codec.http2
            DefaultHttp2DataFrame
            Http2HeadersFrame
            Http2DataFrame]
           [io.netty.handler.codec.http.websocketx.extensions
            WebSocketServerExtension]
           [io.netty.handler.codec.http.websocketx
            WebSocket13FrameDecoder
            WebSocket13FrameEncoder
            WebSocketDecoderConfig]))

(defn websocket-request?
  [req-or-frame]
  (boolean
   (if (instance? Http2HeadersFrame req-or-frame)
     (.contains (.headers ^Http2HeadersFrame req-or-frame) ":protocol" "websocket" true)
     (and (= "websocket" (get-in req-or-frame [:headers ":protocol"]))
          (= "13" (get-in req-or-frame [:headers "sec-websocket-version"]))))))

(defn init-handler
  [^ChannelHandlerContext ctx connection-state ^Http2HeadersFrame msg {:keys [complete-deferred] :as response}]
  (let [stream (.stream msg)
        stream-id (.id stream)
        pipeline (.pipeline ctx)
        {:keys [exec-service]} @connection-state
        ^ExecutorService exec-service exec-service
        {:keys [compression-extension handlers initial-request]} (meta response)
        {:keys [on-open on-close
                on-text-message
                on-binary-message
                on-ping-message
                on-pong-message]} handlers
        max-frame-size (get-in @connection-state [:settings :settings-max-frame-size])
        decoder-config (-> (WebSocketDecoderConfig/newBuilder)
                           (.allowExtensions true)
                           (.maxFramePayloadLength max-frame-size)
                           (.build))
        frame-decoder (WebSocket13FrameDecoder. decoder-config)
        frame-encoder (WebSocket13FrameEncoder. false)
        compression-encoder (when compression-extension
                              (.newExtensionEncoder ^WebSocketServerExtension compression-extension))
        compression-decoder (when compression-extension
                              (.newExtensionDecoder ^WebSocketServerExtension compression-extension))
        closed? (atom false)
        close-stream #(when-let [close (get-in @connection-state [:streams stream-id :close])]
                        (close))
        inbound-handler ^ChannelDuplexHandler (nws/inbound-handler
                                               exec-service
                                               {:on-text-message on-text-message
                                                :on-binary-message on-binary-message
                                                :on-ping-message on-ping-message
                                                :on-pong-message on-pong-message
                                                :on-close on-close})
        handlers (->> [["data-frame-outbound-handler" (proxy [ChannelDuplexHandler] []
                                                        (write [^ChannelHandlerContext ctx
                                                                ^ByteBuf msg
                                                                ^ChannelPromise _p]
                                                          (->> (-> msg
                                                                   (DefaultHttp2DataFrame. false)
                                                                   (.stream stream))
                                                               (n/invoke-write pipeline "frame-writer"))))]
                       ["data-frame-inbound-handler" (proxy [ChannelDuplexHandler] []
                                                       (channelRead [^ChannelHandlerContext ctx ^Http2DataFrame msg]
                                                         (n/acquire msg)
                                                         (let [^ChannelDuplexHandler this this]
                                                           (proxy-super channelRead ctx (.content msg)))))]
                       ["frame-decoder" frame-decoder]
                       ["compression-decoder" compression-decoder]
                       ["frame-encoder" frame-encoder]
                       ["compression-encoder" compression-encoder]
                       ["message-inbound-handler" inbound-handler]
                       ["message-outbound-handler" (proxy [ChannelDuplexHandler] []
                                                     (write [^ChannelHandlerContext ctx
                                                             ^String msg
                                                             ^ChannelPromise p]
                                                       (->> msg
                                                            (nws/data->websocket-frames ctx max-frame-size)
                                                            (nws/send-websocket-frames
                                                             ctx (-> @connection-state
                                                                     :server-args
                                                                     :max-flush-size)))))]]
                      (filter second)
                      (mapv (fn [[handler-name handler]]
                              [(str handler-name "-" stream-id) handler])))

        data-frame-inbound-handler-name (ffirst handlers)
        message-outbound-handler-name (first (last handlers))]
    (swap! (get-in @connection-state [:streams stream-id :cleanup-handlers]) conj
           #(do (reset! closed? true)
                (when on-close
                  (try (on-close)
                       (catch Exception e
                         (println "Exception occurred in on-close handler" e))))
                (doseq [[handler-name _] handlers]
                  (n/safe-remove-handler pipeline handler-name))))
    (doseq [[handler-name handler] handlers]
      (n/safe-add-handler pipeline handler-name handler))
    (when on-open
      (n/run exec-service
        (fn []
          (try
            @complete-deferred
            (on-open
             {:initial-request initial-request
              :send #(when (not @closed?)
                       (n/invoke-write pipeline message-outbound-handler-name %))
              :close (fn []
                       (when (not @closed?)
                         (close-stream)))})
            (catch Exception e
              (.fireExceptionCaught ctx e))))))
    (fn [^Http2DataFrame data-frame]
      (n/invoke-channel-read pipeline data-frame-inbound-handler-name data-frame))))
