;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns utilis.fs
  (:require
   [clojure.java.io :as io]
   [clojure.string :as st]
   [utilis.fn :refer [fsafe]])
  (:import
   [java.io File PrintWriter]
   [java.net URLEncoder]
   [java.nio.file
    AtomicMoveNotSupportedException
    CopyOption
    Files
    Path
    Paths
    StandardCopyOption]
   [java.nio.file.attribute FileAttribute PosixFilePermissions]))

(declare ->file ->path)

(defn working-directory
  []
  (-> (Paths/get "" (into-array String []))
      .toAbsolutePath
      str))

(defn ls
  [path & {:keys [recursive]}]
  (if recursive
    (->> (->file path) file-seq)
    (->> (->file path) (#(.listFiles ^File %)) seq)))

(defn mkdir
  [path & {:keys [recursive]}]
  (if recursive (.mkdirs ^File (->file path)) (.mkdir ^File (->file path))))

(defn rm
  [path & {:keys [recursive]}]
  (if recursive
    (->> (ls path :recursive true) reverse
         (map #(boolean (.delete ^File %)))
         (every? true?))
    (.delete ^File (->file path))))

(defn truncate
  [path]
  (with-open [_ (PrintWriter. (-> ^File (->file path) .getAbsolutePath))]))

(defn cp
  [src dest & {:keys [overwrite copy-attributes]
               :or {overwrite true
                    copy-attributes true}}]
  (let [^Path src (->path src)
        ^Path dest (->path dest)
        options (cond-> #{StandardCopyOption/ATOMIC_MOVE}
                  overwrite (conj StandardCopyOption/REPLACE_EXISTING)
                  copy-attributes (conj StandardCopyOption/COPY_ATTRIBUTES))]
    (Files/copy src dest
                ^"[Ljava.nio.file.StandardCopyOption;"
                (into-array CopyOption options))))

(defn mv
  [src dest & {:keys [overwrite]
               :or {overwrite true}}]
  (let [src (->path src)
        dest (->path dest)
        options (cond-> #{StandardCopyOption/ATOMIC_MOVE}
                  overwrite (conj StandardCopyOption/REPLACE_EXISTING))]
    (try
      (Files/move src dest
                  ^"[Ljava.nio.file.StandardCopyOption;"
                  (into-array CopyOption options))
      (catch AtomicMoveNotSupportedException _
        (Files/move src dest
                    ^"[Ljava.nio.file.StandardCopyOption;"
                    (into-array CopyOption
                                (disj options StandardCopyOption/ATOMIC_MOVE)))))))

(defn directory?
  [path]
  (.isDirectory ^File (->file path)))

(defn file?
  [path]
  (.isFile ^File (->file path)))

(defn mime-type
  [path]
  (let [^File path (->file path)]
    (try (or (->> path (.toPath) (Files/probeContentType))
             (->> path str URLEncoder/encode io/file (.toPath) (Files/probeContentType)))
         (catch java.io.EOFException _ nil))))

(defn image?
  [path]
  (= "image" (first ((fsafe st/split) (mime-type (->file path)) #"/"))))

(defn video?
  [path]
  (= "video" (first ((fsafe st/split) (mime-type (->file path)) #"/"))))

(defn create-temp-dir
  ([prefix]
   (create-temp-dir nil prefix))
  ([path prefix]
   (let [path (if (or (nil? path) (instance? Path path))
                path
                (Paths/get path (into-array String [])))
         _ (mkdir (str path) :recursive true)
         attributes (into-array FileAttribute
                                [(PosixFilePermissions/asFileAttribute
                                  (PosixFilePermissions/fromString "rwx------"))])
         dir (-> (if path
                   (Files/createTempDirectory path prefix attributes)
                   (Files/createTempDirectory prefix attributes))
                 str)]
     (.deleteOnExit (io/file dir))
     dir)))

(defmacro ensure-temp-dir
  "Create temporary directory in os-appropriate location.  Useful when
  body code expects it to be available for file-caching but is running
  in a scratch container.  The temporary directory is deleted when
  exiting scope."
  [& body]
  `(let [temp-dir# (str (java.lang.System/getProperty "java.io.tmpdir")
                        (java.lang.System/getProperty "file.separator")
                        (gensym))]
     ;; Create temporary file location for java.nio to use
     (try
       (mkdir temp-dir# :recursive true)
       ~@body
       (finally (rm temp-dir# :recursive true)))))

(defmacro with-temp
  "bindings => [name [name \"<file-name>\"] ...]
  Evaluates body name(s) bound to temporary java.io.File objects
  in a location appropriate for the operating system. The temporary
  files are deleted when exiting scope."
  [bindings & body]
  (when-not (every? #(or (symbol? %)
                         (and (vector? %)
                              (symbol? (first %))
                              (string? (second %)))) bindings)
    (throw (IllegalArgumentException.
            "with-temp only allows symbols or [symbol \"<file-name>\"] in bindings")))
  (let [temp-dir (gensym)]
    `(ensure-temp-dir
      (let [~temp-dir (java.nio.file.Files/createTempDirectory
                       (str (gensym))
                       (make-array java.nio.file.attribute.FileAttribute 0))]
        (try
          (let ~(->> bindings
                     (map-indexed (fn [i b]
                                    (if (vector? b)
                                      [(first b) `(.toFile (.resolve ~temp-dir ~(second b)))]
                                      [b `(.toFile (.resolve ~temp-dir (str ~i)))])))
                     (reduce concat)
                     vec)
            ~@body)
          (finally (rm (str ~temp-dir) :recursive true)))))))


;;; Private

(defn- ->file ^File
  [path]
  (cond-> path (string? path) io/file))

(defn- ->path ^Path
  [f]
  (-> (->file f)
      .getAbsolutePath
      (Paths/get (into-array [""]))))
