;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns utilis.coll
  (:refer-clojure :exclude [partition-all]))

(defn only
  "Returns the first element from `s` iff it's the only element in `s`.
  If `s` does not contain exactly 1 element, an IllegalAgumentException
  or js/Error is thrown. The message for the exceptions can be optionally
  specified."
  ([s]
   (only s "Sequence does not contain exactly 1 element"))
  ([s exception-message]
   (if (or (not (seq s)) (seq (rest s)))
     (throw #?(:clj (IllegalArgumentException. ^String exception-message)
               :cljs (js/Error exception-message)))
     (first s))))

(defn insert-at
  "Inserts `item` into the `coll` at `index`. Items already in `coll` after
  `index` are shifted forward to accommodate `item`.

  It is the responsibility of the caller to ensure that the index is within
  bounds."
  [coll index item]
  (if (vector? coll)
    (let [len (count coll)]
      (vec (concat (subvec coll 0 index) [item] (subvec coll index len))))
    (concat (take index coll) [item] (drop index coll))))

(defn remove-at
  "Removed the item at `index` from `coll`. Subsequent items are shifted
  to ensure that there are no gaps in the collection.

  It is the responsibility of the caller to ensure that the index is within
  bounds."
  [coll index]
  (if (vector? coll)
    (let [len (count coll)]
      (vec (concat (subvec coll 0 index) (subvec coll (min (inc index) len) len))))
    (concat (take index coll) (drop (inc index) coll))))

(defn nth-last
  "Returns the nth last item in the collection.  Throws an exception if
  the index is out of bounds unless `default` is supplied."
  ([coll index]
   (nth-last coll index ::not-found))
  ([coll index default]
   (let [v (if (vector? coll) coll (vec coll))
         n (count v)]
     (try
       (nth v (- n index 1))
       (catch #?(:clj Throwable :cljs js/Error) e
         (if (not= default ::not-found)
           default
           (throw e)))))))

(defn second-last
  "Returns the second last item in the collection."
  [coll]
  (nth-last coll 1))

(defn llast
  "Returns the last item of the last collection."
  [nested-coll]
  (nth-last (nth-last nested-coll 0) 0))

(defn take-until
  "Returns a lazy sequence of successive items from coll until
  (pred item) returns logical true. The last item, which returned
  true, is included in the output.  pred must be free of
  side-effects.  Transducer not implemented."
  ([pred coll]
   (lazy-seq
    (when-let [s (seq coll)]
      (if (not (pred (first s)))
        (cons (first s) (take-until pred (rest s)))
        (list (first s)))))))

(defn distinct-by
  "Returns a lazy sequence of the elements of coll, removing any elements that
  return duplicate values when passed to a function f. Returns a transducer
  when no collection is provided."
  ([f]
   (fn [rf]
     (let [seen (volatile! #{})]
       (fn
         ([] (rf))
         ([result] (rf result))
         ([result x]
          (let [fx (f x)]
            (if (contains? @seen fx)
              result
              (do (vswap! seen conj fx)
                  (rf result x)))))))))
  ([f coll]
   (let [step (fn step [xs seen]
                (lazy-seq
                 ((fn [[x :as xs] seen]
                    (when-let [s (seq xs)]
                      (let [fx (f x)]
                        (if (contains? seen fx)
                          (recur (rest s) seen)
                          (cons x (step (rest s) (conj seen fx)))))))
                  xs seen)))]
     (step coll #{}))))

(defn key-by
  "Returns a map whose keys are the value returned by the `key-fn`
  applied to each value in the coll.  If the `key-fn` returns a key
  more than once, the last value is used."
  [key-fn coll]
  (persistent!
   (reduce (fn [m v]
             (assoc! m (key-fn v) v))
           (transient {})
           coll)))

(defn partition-all
  "Returns a transducer for the [`n`] and [`n` `step`] arities."
  ([^long n] (partition-all n 1))
  ([^long n ^long step]
   (fn [rf]
     (let [buffer (volatile! [])]
       (fn
         ([] (rf))
         ([result]
          (rf (if (empty? @buffer)
                result
                (let [b @buffer]
                  (vreset! buffer [])
                  (unreduced (rf result b))))))
         ([result v]
          (let [b (vswap! buffer conj v)]
            (if (= n (count b))
              (do
                (vswap! buffer (comp vec (partial drop step)))
                (rf result b))
              result)))))))
  ([n step coll]
   (lazy-seq
    (when-let [s (seq coll)]
      (let [seg (doall (take n s))]
        (cons seg (partition-all n step (nthrest s step))))))))
