;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns utilis.inflections
  (:refer-clojure :exclude [name])
  (:require [clojure.string :as st]))

(declare uncountable-words name normalize transform coerce
         plural-rules singular-rules)

(defn uncountable?
  [x]
  (boolean (get uncountable-words (normalize x))))

(def countable? (complement uncountable?))

(defn plural
  [x]
  (transform plural-rules x))

(defn singular
  [x]
  (transform singular-rules x))

(defn hyphenate
  [x]
  (coerce x (some-> (name x)
                    (st/replace #"([A-Z]+)([A-Z][a-z])" "$1-$2")
                    (st/replace #"([a-z\d])([A-Z])" "$1-$2")
                    (st/replace #"\s+" "-")
                    (st/replace #"_" "-")
                    st/lower-case)))

(defn underscore
  [x]
  (coerce x (some-> (name x)
                    (st/replace #"([A-Z\d]+)([A-Z][a-z])" "$1_$2")
                    (st/replace #"([a-z\d])([A-Z])" "$1_$2")
                    (st/replace #"-" "_")
                    st/lower-case)))

(defn camel-case
  [word & [mode]]
  (when word
    (coerce word
            (let [camel-case (-> (st/replace (name word) #"/(.?)" #(str "/" (st/upper-case (nth % 1))))
                                 (st/replace #"(^|_|-)(.)"
                                             #?(:clj
                                                #(str (when (#{\_ \-} (nth % 1))
                                                        (nth % 1))
                                                      (st/upper-case (nth % 2)))
                                                :cljs
                                                #(let [[_ _ letter-to-uppercase] %]
                                                   (st/upper-case letter-to-uppercase)))))]
              (if (= :lower mode)
                (let [first-char (cond-> (first camel-case) (= :lower mode) st/lower-case)]
                  (apply str (cons first-char (rest camel-case))))
                camel-case)))))

(defn capitalize
  [word]
  (when word
    (let [normalized (name word)]
      (->> (str (st/upper-case (str (first normalized)))
                (when (next normalized)
                  (st/lower-case (subs normalized 1))))
           (coerce word)))))

(defn titleize
  [s]
  (when s
    (->> (st/split (name s) #"[-_./ ]")
         (map capitalize)
         (st/join " ")
         (coerce s))))


;;; Private

(defn- name
  "Preserves namespace for keywords and symbols, unlike `clojure.core/name`"
  [x]
  (cond
    (nil? x) x
    (string? x) x
    (or (keyword? x) (symbol? x)) (if-let [ns (namespace x)]
                                    (str ns "/" (clojure.core/name x))
                                    (clojure.core/name x))))

(defn- normalize
  [x]
  (st/lower-case (name x)))

(defn- apply-rules
  [rules word]
  (->> rules reverse
       (map (fn [[pattern replacement]]
              (when (re-find pattern word)
                (st/replace word pattern replacement))))
       (remove nil?)
       first))

(defn- coerce
  [original result]
  (cond-> result
    (keyword? original) keyword
    (symbol? original) symbol))

(defn- transform
  [rules x]
  (let [normalized-s (name x)]
    (if (or (st/blank? normalized-s)
            (uncountable? normalized-s))
      x
      (coerce x (apply-rules rules normalized-s)))))

(def ^:private uncountable-words
  #{"air" "alcohol" "art" "blood" "butter" "cheese" "chewing" "coffee"
    "confusion" "cotton" "education" "electricity" "entertainment" "equipment"
    "experience" "fiction" "fish" "food" "forgiveness" "fresh" "gold" "gossip" "grass"
    "ground" "gum" "happiness" "history" "homework" "honey" "ice" "information" "jam"
    "knowledge" "lightning" "liquid" "literature" "love" "luck" "luggage" "meat" "milk"
    "mist" "money" "music" "news" "oil" "oxygen" "paper" "patience" "peanut" "pepper"
    "petrol" "pork" "power" "pressure" "research" "rice" "sadness" "series" "sheep"
    "shopping" "silver" "snow" "space" "species" "speed" "steam" "sugar" "sunshine" "tea"
    "tennis" "thunder" "time" "toothpaste" "traffic" "up" "vinegar" "washing" "wine"
    "wood" "wool"})

(def ^:private plural-rules
  [[#"(?i)$" "s"]
   [#"(?i)s$" "s"]
   [#"(?i)(ax|test)is$" "$1es"]
   [#"(?i)(octop|vir)us$" "$1i"]
   [#"(?i)(alias|status)$" "$1es"]
   [#"(?i)(bu)s$" "$1ses"]
   [#"(?i)(buffal|tomat)o$" "$1oes"]
   [#"(?i)([ti])um$" "$1a"]
   [#"(?i)sis$" "ses"]
   [#"(?i)(?:([^f])fe|([lr])f)$" "$1$2ves"]
   [#"(?i)(hive)$" "$1s"]
   [#"(?i)([^aeiouy]|qu)y$" "$1ies"]
   [#"(?i)(x|ch|ss|sh)$" "$1es"]
   [#"(?i)(matr|vert|ind)(?:ix|ex)$" "$1ices"]
   [#"(?i)([m|l])ouse$" "$1ice"]
   [#"(?i)^(ox)$" "$1en"]
   [#"(?i)(quiz)$" "$1zes"]])

(def ^:private singular-rules
  [[#"(?i)s$" ""]
   [#"(?i)(ss)$" "$1"]
   [#"(?i)(n)ews$" "$1ews"]
   [#"(?i)([ti])a$" "$1um"]
   [#"(?i)((a)naly|(b)a|(d)iagno|(p)arenthe|(p)rogno|(s)ynop|(t)he)(sis|ses)$" "$1$2sis"]
   [#"(?i)(^analy)(sis|ses)$" "$1sis"]
   [#"(?i)([^f])ves$" "$1fe"]
   [#"(?i)(hive)s$" "$1"]
   [#"(?i)(tive)s$" "$1"]
   [#"(?i)([lr])ves$" "$1f"]
   [#"(?i)([^aeiouy]|qu)ies$" "$1y"]
   [#"(?i)(s)eries$" "$1eries"]
   [#"(?i)(m)ovies$" "$1ovie"]
   [#"(?i)(x|ch|ss|sh)es$" "$1"]
   [#"(?i)([m|l])ice$" "$1ouse"]
   [#"(?i)(bus)(es)?$" "$1"]
   [#"(?i)(o)es$" "$1"]
   [#"(?i)(shoe)s$" "$1"]
   [#"(?i)(cris|ax|test)(is|es)$" "$1is"]
   [#"(?i)(octop|vir)(us|i)$" "$1us"]
   [#"(?i)(alias|status)(es)?$" "$1"]
   [#"(?i)^(ox)en" "$1"]
   [#"(?i)(vert|ind)ices$" "$1ex"]
   [#"(?i)(matr)ices$" "$1ix"]
   [#"(?i)(quiz)zes$" "$1"]
   [#"(?i)(database)s$" "$1"]])
