;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns utilis.coll)

(defn only
  "Returns the first element from `s` iff it's the only element in `s`.
  If `s` does not contain exactly 1 element, an IllegalAgumentException
  or js/Error is thrown. The message for the exceptions can be optionally
  specified."
  ([s]
   (only s "Sequence does not contain exactly 1 element"))
  ([s exception-message]
   (if (or (not (seq s)) (seq (rest s)))
     (throw #?(:clj (IllegalArgumentException. ^String exception-message)
               :cljs (js/Error exception-message)))
     (first s))))

(defn insert-at
  "Inserts `item` into the `coll` at `index`. Items already in `coll` after
  `index` are shifted forward to accommodate `item`.

  It is the responsibility of the caller to ensure that the index is within
  bounds."
  [coll index item]
  (if (vector? coll)
    (let [len (count coll)]
      (vec (concat (subvec coll 0 index) [item] (subvec coll index len))))
    (concat (take index coll) [item] (drop index coll))))

(defn remove-at
  "Removed the item at `index` from `coll`. Subsequent items are shifted
  to ensure that there are no gaps in the collection.

  It is the responsibility of the caller to ensure that the index is within
  bounds."
  [coll index]
  (if (vector? coll)
    (let [len (count coll)]
      (vec (concat (subvec coll 0 index) (subvec coll (min (inc index) len) len))))
    (concat (take index coll) (drop (inc index) coll))))

(defn nth-last
  "Returns the nth last item in the collection.  Throws an exception if
  the index is out of bounds unless `default` is supplied."
  ([coll index]
   (nth-last coll index ::not-found))
  ([coll index default]
   (let [v (if (vector? coll) coll (vec coll))
         n (count v)]
     (try
       (nth v (- n index 1))
       (catch #?(:clj Throwable :cljs js/Error) e
         (if (not= default ::not-found)
           default
           (throw e)))))))

(defn second-last
  "Returns the second last item in the collection."
  [coll]
  (nth-last coll 1))

(defn llast
  "Returns the last item of the last collection."
  [nested-coll]
  (nth-last (nth-last nested-coll 0) 0))

(defn take-until
  "Returns a lazy sequence of successive items from coll until
  (pred item) returns logical true. The last item, which returned
  true, is included in the output.  pred must be free of
  side-effects.  Transducer not implemented."
  ([pred coll]
   (lazy-seq
    (when-let [s (seq coll)]
      (if (not (pred (first s)))
        (cons (first s) (take-until pred (rest s)))
        (list (first s)))))))
