;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns signum.subs
  (:refer-clojure :exclude [namespace subs])
  (:require [signum.signal :refer [with-tracking] :as s]
            [signum.interceptors :refer [->interceptor] :as interceptors]
            [signum.fx :as fx]
            #?(:cljs [signum.SignalColdException :refer [SignalColdException]])
            [utilis.map :refer [compact]]
            [utilis.coll :refer [remove-at]]
            [clojure.set :as set])
  #?(:clj (:import [signum SignalColdException])))

(defonce ^:dynamic *context* {})
(defonce ^:dynamic *current-sub-fn* nil)

(declare handlers subscriptions create-subscription! dispose-subscription! signal-interceptor)

(defn reg-sub
  [query-id & args]
  (when-not (fn? (last args))
    (throw (ex-info "computation-fn must be provided"
                    {:query-id query-id :args args})))
  (let [options (when (map? (first args)) (first args))
        interceptors (or (when (vector? (first args)) (first args))
                         (when (and (map? (first args))
                                    (vector? (second args)))
                           (second args)))
        fns (filter fn? args)
        [init-fn dispose-fn computation-fn] (case (count fns)
                                              3 fns
                                              2 [(first fns) nil (last fns)]
                                              1 [nil nil (last fns)])
        sub (assoc
             (compact
              {:init-fn init-fn
               :dispose-fn dispose-fn
               :computation-fn computation-fn
               :options options
               :queue (vec (concat [fx/interceptor]
                                   interceptors
                                   [signal-interceptor]))
               :limbo (let [max (get-in options [:limbo :max-count] 0)]
                        (atom {:queue (let [[keep dispose] (split-at max (get-in @handlers [query-id :limbo :queue]))]
                                        (doseq [[output-signal {:keys [query-v]}] dispose]
                                          (dispose-subscription! query-v output-signal))
                                        keep)
                               :max max}))
               :ns *ns*})
             :stack [])]
    (doseq [[output-signal {:keys [query-v context]}] @subscriptions
            :when (= query-id (first query-v))]
      (dispose-subscription! query-v output-signal)
      (binding [*context* context]
        (create-subscription! query-v output-signal)))
    (swap! handlers assoc query-id sub))
  query-id)

(defn subscribe
  [[query-id & _ :as query-v]]
  (when (= ::init-fn *current-sub-fn*)
    (throw (ex-info "signum.subs/subscribe is not supported within the init-fn of a sub"
                    {:query-v query-v})))
  (when (= ::dispose-fn *current-sub-fn*)
    (throw (ex-info "signum.subs/subscribe is not supported within the dispose-fn of a sub"
                    {:query-v query-v})))
  (if-let [handler-context (get @handlers query-id)]
    (-> (merge *context* handler-context)
        (assoc-in [:coeffects :query-v] query-v)
        interceptors/run
        (get-in [:effects :signal]))
    #?(:clj (throw (ex-info "invalid query" {:query query-v}))
       :cljs (throw (js/Error. (str "invalid query: " {:query query-v}))))))

(defn interceptors
  [query-id]
  (get-in @handlers [query-id :queue]))

(defn namespace
  [id]
  (get-in @handlers [id :ns]))

(declare signals)

(defn sub?
  [id]
  (contains? @handlers id))

(defn subs
  []
  (keys @signals))

(defn trace
  [query-id trace-id trace-fn]
  (swap! handlers assoc-in [query-id :tracers trace-id] trace-fn))

(defn un-trace
  [query-id trace-id]
  (swap! handlers update-in [query-id :tracers] dissoc trace-id))


;;; Private

(defonce ^:private handlers (atom {}))      ; key: query-id
(defonce ^:private signals (atom {}))       ; key: query-v
(defonce ^:private subscriptions (atom {})) ; key: output-signal

(defn- create-subscription!
  [query-v output-signal]
  (let [query-id (first query-v)
        handlers (get @handlers query-id)
        {:keys [init-fn computation-fn]} handlers
        init-context (when init-fn
                       (try
                         (binding [*current-sub-fn* ::init-fn]
                           (init-fn query-v))
                         (catch #?(:clj Exception :cljs :default) e
                           e)))
        input-signals (atom #{})
        compute-queue-size (atom 0)
        compute (fn compute [current-value]
                  (if (zero? (swap! compute-queue-size dec))
                    (let [derefed (atom #{})
                          track (fn [s]
                                  (when-not (or (get @input-signals s)
                                                (get @derefed s))
                                    (swap! input-signals conj s)
                                    (add-watch s query-v
                                               (fn [_ _ old-value new-value]
                                                 (when (not= old-value new-value)
                                                   (swap! compute-queue-size inc)
                                                   (s/alter! output-signal compute)))))
                                  (swap! derefed conj s))]
                      (try
                        (binding [*current-sub-fn* ::compute-fn]
                          (with-tracking
                            (fn [reason s]
                              (when (= :deref reason)
                                (track s)))
                            (let [new-value (if init-fn
                                              (computation-fn init-context query-v)
                                              (computation-fn query-v))]
                              (when-let [tracers (not-empty (:tracers handlers))]
                                (doseq [[trace-id trace-fn] tracers]
                                  (trace-fn trace-id query-v current-value new-value)))
                              new-value)))
                        (catch SignalColdException e
                          (when-let [s (some-> e ex-data :signal)]
                            (track s))
                          :signum.signal/cold)
                        (catch #?(:clj Exception :cljs :default) e e)
                        (finally
                          (doseq [w (set/difference @input-signals @derefed)]
                            (remove-watch w query-v))
                          (reset! input-signals @derefed))))
                    current-value))]
    (swap! compute-queue-size inc)
    (s/alter! output-signal compute)
    (swap! subscriptions assoc output-signal (compact
                                              {:query-v query-v
                                               :context *context*
                                               :handlers handlers
                                               :init-context init-context
                                               :input-signals input-signals}))
    output-signal))

(defn- dispose-subscription!
  [query-v output-signal]
  (try
    (when-let [subscription (get @subscriptions output-signal)]
      (let [{:keys [init-context handlers input-signals]} subscription]
        (doseq [w @input-signals] (remove-watch w query-v))
        (when-let [dispose-fn (:dispose-fn handlers)]
          (binding [*current-sub-fn* ::dispose-fn]
            (dispose-fn init-context query-v)))))
    (catch #?(:clj Exception :cljs :default) _)
    (finally
      (swap! subscriptions dissoc output-signal)
      (s/alter! output-signal (constantly :signum.signal/cold)))))

(defn- signal
  [query-v]
  (locking signals
    (let [hashed-key (hash query-v)]
      (or (get @signals hashed-key)
          (let [query-id (first query-v)
                output-signal (with-meta (s/signal) {:signum/query-v query-v})]
            (s/add-watcher-watch output-signal hashed-key
                                 (fn [_ _ old-watchers new-watchers]
                                   (let [old-count (count old-watchers)
                                         new-count (count new-watchers)
                                         limbo (get-in @handlers [query-id :limbo])
                                         {:keys [queue max]} @limbo
                                         limbo-create
                                         (fn [subscription]
                                           (let [index (->> queue
                                                            (map-indexed (fn [i s] (when (= s subscription) i)))
                                                            (remove nil?)
                                                            first)]
                                             (if index
                                               (swap! limbo update :queue remove-at index)
                                               (apply create-subscription! subscription))))
                                         limbo-dispose
                                         (fn [subscription]
                                           (if (zero? max)
                                             (apply dispose-subscription! subscription)
                                             (if (< (count queue) max)
                                               (swap! limbo update :queue (partial cons subscription))
                                               (do (apply dispose-subscription! (last queue))
                                                   (swap! limbo update :queue (comp (partial cons subscription) drop-last))))))]
                                     (cond
                                       (and (zero? old-count) (pos? new-count))
                                       (limbo-create [query-v output-signal])

                                       (and (pos? old-count) (= 0 new-count))
                                       (limbo-dispose [query-v output-signal])

                                       :else nil))))
            (swap! signals assoc hashed-key output-signal)
            output-signal)))))

(def ^:private signal-interceptor
  (->interceptor
   :id :signum.subs/signal-interceptor
   :before #(assoc-in % [:effects :signal] (signal (get-in % [:coeffects :query-v])))))
