;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns signum.subs
  (:refer-clojure :exclude [namespace subs])
  (:require [signum.signal :refer [with-tracking] :as s]
            [signum.interceptors :refer [->interceptor] :as interceptors]
            [signum.fx :as fx]
            #?(:cljs [signum.SignalColdException :refer [SignalColdException]])
            [utilis.map :refer [compact]]
            [clojure.set :as set])
  #?(:clj (:import [signum SignalColdException])))

(defonce ^:dynamic *context* {})
(defonce ^:dynamic *current-sub-fn* nil)

(declare handlers reset-subscriptions! release! signal-interceptor)

(defn reg-sub
  [query-id & args]
  (when-not (fn? (last args))
    (throw (ex-info "computation-fn must be provided"
                    {:query-id query-id :args args})))
  (let [options (when (map? (first args)) (first args))
        interceptors (or (when (vector? (first args)) (first args))
                         (when (and (map? (first args))
                                    (vector? (second args)))
                           (second args)))
        fns (filter fn? args)
        [init-fn dispose-fn computation-fn] (case (count fns)
                                              3 fns
                                              2 [(first fns) nil (last fns)]
                                              1 [nil nil (last fns)])]
    (swap! handlers assoc query-id
           (assoc
            (compact
             {:init-fn init-fn
              :dispose-fn dispose-fn
              :computation-fn computation-fn
              :options options
              :queue (vec (concat [fx/interceptor]
                                  interceptors
                                  [signal-interceptor]))
              :ns *ns*})
            :stack [])))
  (reset-subscriptions! query-id)
  query-id)

(defn subscribe
  [[query-id & _ :as query-v]]
  (when (= ::init-fn *current-sub-fn*)
    (throw (ex-info "signum.subs/subscribe is not supported within the init-fn of a sub"
                    {:query-v query-v})))
  (when (= ::dispose-fn *current-sub-fn*)
    (throw (ex-info "signum.subs/subscribe is not supported within the dispose-fn of a sub"
                    {:query-v query-v})))
  (if-let [handler-context (get @handlers query-id)]
    (-> (merge *context* handler-context)
        (assoc-in [:coeffects :query-v] query-v)
        interceptors/run
        (get-in [:effects :signal]))
    (throw (ex-info "invalid query" {:query query-v}))))

(defn interceptors
  [query-id]
  (get-in @handlers [query-id :queue]))

(defn namespace
  [id]
  (get-in @handlers [id :ns]))

(declare signals)

(defn sub?
  [id]
  (contains? @handlers id))

(defn subs
  []
  (keys @signals))


;;; Private

(defonce ^:private handlers (atom {}))      ; key: query-id
(defonce ^:private signals (atom {}))       ; key: query-v
(defonce ^:private subscriptions (atom {})) ; key: output-signal
(defonce ^:private limbo (atom {}))         ; key: query-id

(defn- create-subscription!
  [query-v output-signal]
  (let [query-id (first query-v)
        handlers (get @handlers query-id)
        {:keys [init-fn computation-fn]} handlers
        init-context (when init-fn
                       (try
                         (binding [*current-sub-fn* ::init-fn]
                           (init-fn query-v))
                         (catch #?(:clj Exception :cljs :default) e
                           e)))
        input-signals (atom #{})
        compute-queue-size (atom 0)
        compute (fn compute [current-value]
                  (if (zero? (swap! compute-queue-size dec))
                    (let [derefed (atom #{})
                          track (fn [s]
                                  (when-not (or (get @input-signals s)
                                                (get @derefed s))
                                    (swap! input-signals conj s)
                                    (add-watch s query-v
                                               (fn [_ _ old-value new-value]
                                                 (when (not= old-value new-value)
                                                   (swap! compute-queue-size inc)
                                                   (s/alter! output-signal compute)))))
                                  (swap! derefed conj s))]
                      (try
                        (binding [*current-sub-fn* ::compute-fn]
                          (with-tracking
                            (fn [reason s]
                              (when (= :deref reason)
                                (track s)))
                            (if init-fn
                              (computation-fn init-context query-v)
                              (computation-fn query-v))))
                        (catch SignalColdException e
                          (when-let [s (some-> e ex-data :signal)]
                            (track s))
                          :signum.signal/cold)
                        (catch #?(:clj Exception :cljs :default) e e)
                        (finally
                          (doseq [w (set/difference @input-signals @derefed)]
                            (remove-watch w query-v))
                          (reset! input-signals @derefed))))
                    current-value))]
    (swap! compute-queue-size inc)
    (s/alter! output-signal compute)
    (swap! subscriptions assoc output-signal (compact
                                              {:query-v query-v
                                               :context *context*
                                               :handlers handlers
                                               :init-context init-context
                                               :input-signals input-signals}))
    (when-let [limbo-max (-> handlers :options :limbo :max-count)]
      (when (pos? limbo-max)
        (when (= limbo-max (count (get @limbo query-id)))
          (apply remove-watch (last (get @limbo query-id)))
          (swap! limbo update query-id drop-last))
        (let [watch-key [:limbo query-v]]
          (add-watch output-signal watch-key (fn [_ _ _ _]))
          (swap! limbo update query-id
                 (partial cons [output-signal watch-key])))))
    output-signal))

(defn- dispose-subscription!
  [query-v output-signal]
  (when-let [subscription (get @subscriptions output-signal)]
    (let [{:keys [init-context handlers input-signals]} subscription]
      (doseq [w @input-signals] (remove-watch w query-v))
      (when-let [dispose-fn (:dispose-fn handlers)]
        (try
          (binding [*current-sub-fn* ::dispose-fn]
            (dispose-fn init-context query-v))
          (s/alter! output-signal (constantly :signum.signal/cold))
          (catch #?(:clj Exception :cljs js/Error) e
            (s/alter! output-signal (constantly e)))))
      (swap! subscriptions dissoc output-signal)
      nil)))

(defn- reset-subscriptions!
  [query-id]
  (locking signals
    (doseq [[query-v output-signal] (filter (fn [[query-v _]]
                                              (= query-id (first query-v))) @signals)]
      (when (get @subscriptions output-signal)
        (let [context (get-in @subscriptions [output-signal :context])]
          (dispose-subscription! query-v output-signal)
          (binding [*context* context]
            (create-subscription! query-v output-signal)))))))

(defn- handle-watches
  [query-v signal _ _ old-watchers watchers]
  (locking signals
    (cond
      (and (= 0 (count old-watchers))
           (= 1 (count watchers))) (create-subscription! query-v signal)
      (and (pos? (count old-watchers))
           (= 0 (count watchers))) (dispose-subscription! query-v signal)
      :else nil)))

(defn- signal
  [query-v]
  (locking signals
    (or (get @signals query-v)
        (let [output-signal (with-meta (s/signal) {:signum/query-v query-v})]
          (s/add-watcher-watch output-signal
                               query-v
                               (partial handle-watches query-v output-signal))
          (swap! signals assoc query-v output-signal)
          output-signal))))

(def ^:private signal-interceptor
  (->interceptor
   :id :signum.subs/signal-interceptor
   :before #(assoc-in % [:effects :signal] (signal (get-in % [:coeffects :query-v])))))
