;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns signum.debounce
  (:require [signum.events :as se]))

(def ^:private deferred-actions (atom {}))

(defn dispatch-debounce
  [dispatch-map-or-seq]
  (let [cancel-timeout (fn [id]
                         (when-let [deferred (get @deferred-actions id)]
                           (js/clearTimeout (:timer deferred))
                           (swap! deferred-actions dissoc id)))
        run-action (fn [action event]
                     (cond
                       (= :dispatch action) (se/dispatch nil event)
                       (= :dispatch-n action) (doseq [e event]
                                                (se/dispatch nil e))))]
    (doseq [{:keys [id timeout action event]}
            (cond-> dispatch-map-or-seq
              (not (sequential? dispatch-map-or-seq)) vector)]
      (cond
        (#{:dispatch :dispatch-n} action)
        (do (cancel-timeout id)
            (swap! deferred-actions assoc id
                   {:action action
                    :event event
                    :timer (js/setTimeout (fn []
                                            (cancel-timeout id)
                                            (run-action action event))
                                          timeout)}))

        (= :cancel action)
        (cancel-timeout id)

        (= :flush action)
        (when-let [{:keys [action event]} (get @deferred-actions id)]
          (cancel-timeout id)
          (run-action action event))

        :else
        (throw (js/Error (str ":dispatch-debounce invalid action " action)))))))
