;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns signum.subs
  (:refer-clojure :exclude [namespace subs])
  (:require
   #?(:cljs [signum.SignalColdException :refer [SignalColdException]])
   #?(:cljs [signum.NoOpException :refer [NoOpException]])
   [clojure.set :as set]
   [fluxus.lock :refer [with-lock]]
   [signum.fx :as fx]
   [signum.interceptors :refer [->interceptor] :as interceptors]
   [signum.signal :as s]
   [utilis.coll :refer [remove-at]]
   [utilis.map :refer [compact]])
  #?(:clj (:import
           [java.util.concurrent.locks ReentrantLock]
           [signum NoOpException SignalColdException])))

(defonce ^:dynamic *context* {})
(defonce ^:dynamic *current-sub-fn* nil)

(declare handlers subscriptions tracers
         create-subscription! dispose-subscription! signal-interceptor)

(defn reg-sub
  [query-id & args]
  (when-not (fn? (last args))
    (throw (ex-info ":signum.subs/reg-sub computation-fn must be provided"
                    {:query-id query-id :args args})))
  (let [options (when (map? (first args)) (first args))
        interceptors (or (when (vector? (first args)) (first args))
                         (when (and (map? (first args))
                                    (vector? (second args)))
                           (second args)))
        fns (filter fn? args)
        [init-fn dispose-fn computation-fn] (case (count fns)
                                              3 fns
                                              2 [(first fns) nil (last fns)]
                                              1 [nil nil (last fns)])
        sub (assoc
             (compact
              {:init-fn init-fn
               :dispose-fn dispose-fn
               :computation-fn computation-fn
               :options options
               :queue (vec (concat [fx/interceptor]
                                   interceptors
                                   [signal-interceptor]))
               :cache (let [max (get-in options [:cache :max] 0)]
                        (atom {:queue (let [[keep dispose] (split-at max (get-in @handlers [query-id :cache :queue]))]
                                        (doseq [[output-signal {:keys [query-v]}] dispose]
                                          (dispose-subscription! query-v output-signal))
                                        keep)
                               :max max}))
               :ns *ns*})
             :stack [])
        active (filter (fn [[_ {:keys [query-v]}]]
                         (= query-id (first query-v))) @subscriptions)]
    (doseq [[output-signal {:keys [query-v]}] active]
      (dispose-subscription! query-v output-signal :set-cold false))
    (swap! handlers assoc query-id sub)
    (doseq [[output-signal {:keys [query-v context]}] active]
      (binding [*context* context]
        (create-subscription! query-v output-signal))))
  query-id)

(defn dereg-sub
  [query-id]
  (let [active (filter (fn [[_ {:keys [query-v]}]]
                         (= query-id (first query-v))) @subscriptions)]
    (doseq [[output-signal {:keys [query-v]}] active]
      (dispose-subscription! query-v output-signal))
    (swap! handlers dissoc query-id)
    nil))

(defn subscribe
  [[query-id & _ :as query-v]]
  (when (= ::init-fn *current-sub-fn*)
    (throw (ex-info "signum.subs/subscribe is not supported within the init-fn of a sub"
                    {:query-v query-v})))
  (when (= ::dispose-fn *current-sub-fn*)
    (throw (ex-info "signum.subs/subscribe is not supported within the dispose-fn of a sub"
                    {:query-v query-v})))
  (if-let [handler-context (get @handlers query-id)]
    (-> (merge *context* handler-context)
        (assoc-in [:coeffects :query-v] query-v)
        interceptors/run
        (get-in [:effects :signal]))
    #?(:clj (throw (ex-info ":signum/subscribe invalid query" {:query query-v}))
       :cljs (throw (js/Error. (str ":signum/subscribe invalid query: " {:query query-v}))))))

(defn interceptors
  [query-id]
  (get-in @handlers [query-id :queue]))

(defn namespace
  [id]
  (get-in @handlers [id :ns]))

(declare signals)

(defn sub?
  [id]
  (contains? @handlers id))

(defn subs
  []
  (keys @signals))

(defn no-op
  []
  (throw (NoOpException.)))

(defn trace
  [query-id trace-id trace-fn]
  (when (= ::unknown (get @handlers query-id ::unknown))
    (throw (ex-info ":signum.subs/trace unknown query-id" {:query-id query-id})))
  (swap! tracers assoc-in [query-id trace-id] trace-fn)
  query-id)

(defn detrace
  [query-id trace-id]
  (swap! tracers update query-id dissoc trace-id)
  query-id)

(defn trace-all
  [trace-id trace-fn]
  (doseq [query-id (keys @handlers)]
    (trace query-id trace-id trace-fn)))

(defn detrace-all
  [trace-id]
  (doseq [query-id (keys @handlers)]
    (detrace query-id trace-id)))


;;; Private

(defonce ^:private handlers (atom {}))      ; key: query-id
(defonce ^:private signals (atom {}))       ; key: query-v
(defonce ^:private subscriptions (atom {})) ; key: output-signal
(defonce ^:private tracers (atom {}))       ; key: query-id
(defonce ^:private signal-lock #?(:clj (ReentrantLock.) :cljs nil))

(defn- handle-tracers
  [query-v event value]
  (when-let [tracers (not-empty (get @tracers (first query-v)))]
    (doseq [[trace-id trace-fn] tracers]
      (try
        (trace-fn trace-id query-v event value)
        (catch #?(:clj Exception :cljs :default) _)))))

(defn- create-subscription!
  [query-v output-signal]
  (let [query-id (first query-v)
        handlers (get @handlers query-id)
        {:keys [init-fn computation-fn]} handlers
        init-context (when init-fn
                       (try
                         (binding [*current-sub-fn* ::init-fn]
                           (handle-tracers query-v :init output-signal)
                           (init-fn query-v))
                         (catch #?(:clj Exception :cljs :default) e
                           e)))
        input-signals (atom #{})
        compute (fn compute [current-value]
                  (let [derefed (atom #{})
                        track (fn [s]
                                (swap! derefed conj s)
                                (when-not (get @input-signals s)
                                  (swap! input-signals conj s)
                                  (add-watch
                                   s query-v
                                   (fn [_ _ old-value new-value]
                                     (when (not= old-value new-value)
                                       (s/alter! output-signal compute))))))]
                    (try
                      (binding [*current-sub-fn* ::compute-fn]
                        (s/with-tracer
                            (fn [event s]
                              (when (= :deref event)
                                (track s)))
                          (try
                            (let [new-value (if init-fn
                                              (computation-fn init-context query-v)
                                              (computation-fn query-v))]
                              (handle-tracers query-v :value [current-value new-value])
                              new-value)
                            (catch NoOpException _
                              (reset! input-signals @derefed)
                              current-value))))
                      (catch SignalColdException e
                        (when-let [s (some-> e ex-data :signal)]
                          (track s)
                          (handle-tracers query-v :cold-dep [(some-> s meta :signum/query-v)]))
                        s/cold-value)
                      (catch #?(:clj Exception :cljs :default) e
                        (handle-tracers query-v :error e)
                        e)
                      (finally
                        (doseq [w (set/difference @input-signals @derefed)]
                          (remove-watch w query-v))
                        (reset! input-signals @derefed)))))]
    (s/alter! output-signal compute)
    (swap! subscriptions assoc output-signal {:query-v query-v
                                              :context *context*
                                              :init-context init-context
                                              :input-signals input-signals})
    output-signal))

(defn- dispose-subscription!
  [query-v output-signal & {:keys [set-cold] :or {set-cold true}}]
  (try
    (when-let [subscription (get @subscriptions output-signal)]
      (let [handlers (get @handlers (first query-v))
            {:keys [init-context input-signals]} subscription]
        (doseq [w @input-signals] (remove-watch w query-v))
        (when-let [dispose-fn (:dispose-fn handlers)]
          (binding [*current-sub-fn* ::dispose-fn]
            (dispose-fn init-context query-v)))))
    (catch #?(:clj Exception :cljs :default) e
      (handle-tracers query-v :error e))
    (finally
      (swap! subscriptions dissoc output-signal)
      (when set-cold (s/alter! output-signal (constantly s/cold-value)))
      (handle-tracers query-v :dispose output-signal))))

(defn- signal
  [query-v]
  (with-lock [signal-lock]
    (or (get @signals query-v)
        (let [query-id (first query-v)
              output-signal (with-meta (s/signal s/cold-value [:dropping 1])
                              {:signum/query-v query-v})]
          (s/add-watcher-watch
           output-signal
           query-v
           (fn [_ _ old-watchers new-watchers]
             (let [old-count (count old-watchers)
                   new-count (count new-watchers)
                   cache (get-in @handlers [query-id :cache])
                   {:keys [queue max]} @cache
                   cache-create
                   (fn [subscription]
                     (let [index (->> queue
                                      (map-indexed (fn [i s] (when (= s subscription) i)))
                                      (remove nil?)
                                      first)]
                       (if index
                         (swap! cache update :queue remove-at index)
                         (apply create-subscription! subscription))))
                   cache-dispose
                   (fn [subscription]
                     (if (zero? max)
                       (apply dispose-subscription! subscription)
                       (if (< (count queue) max)
                         (swap! cache update :queue (partial cons subscription))
                         (do (apply dispose-subscription! (last queue))
                             (swap! cache update :queue (comp (partial cons subscription) drop-last))))))]
               (cond
                 (and (zero? old-count) (pos? new-count))
                 (cache-create [query-v output-signal])

                 (and (pos? old-count) (zero? new-count))
                 (cache-dispose [query-v output-signal])

                 :else nil))))
          (swap! signals assoc query-v output-signal)
          output-signal))))

(def ^:private signal-interceptor
  (->interceptor
   :id :signum.subs/signal-interceptor
   :before #(assoc-in % [:effects :signal] (signal (get-in % [:coeffects :query-v])))))
