;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns signum.signal
  #?(:cljs (:require-macros [signum.signal]))
  (:require
   #?(:cljs [signum.SignalColdException :refer [SignalColdException]])
   #?(:cljs [utilis.js :as j])
   #?(:clj [clojure.pprint :refer [simple-dispatch]])
   [fluxus.flow :as f]
   [fluxus.lock :refer [with-lock]]
   [fluxus.promise :as p]
   [utilis.fn :refer [fsafe]]
   [utilis.string :as ust])
  #?(:clj
     (:import
      [clojure.lang
       IDeref
       IMeta
       IObj
       IRef]
      [java.util UUID]
      [java.util.concurrent.locks ReentrantLock]
      [signum SignalColdException])))

(def cold-value :signum.signal/cold)

(def ^:dynamic *tracer* nil)

(declare pr-signal)

(defprotocol IWatchWatchers
  (add-watcher-watch [signal key watch-fn])
  (remove-watcher-watch [signal key])
  (watchers [signal]))

(defprotocol IState
  (hot? [signal])
  (cold? [signal])
  (error? [signal]))

(defprotocol IAlter
  (-alter! [signal fun args]))

(defn alter! [signal fun & args] (-alter! signal fun args))

(deftype Signal [backend watches alter-queue alter-lock buffer-spec meta-map]

  Object
  (toString [^Signal this]
    (pr-signal this))
  #?(:cljs IHash)
  (#?(:clj hashCode :cljs -hash) [_]
    (hash [:signum/signal backend]))
  #?(:cljs IEquiv)
  (#?(:clj equals :cljs -equiv) [this other]
    (= (#?(:clj .backend :cljs .-backend) this)
       (#?(:clj .backend :cljs .-backend) ^Signal other)))

  IState
  (hot? [this]
    (not (or (cold? this) (error? this))))
  (cold? [_this]
    (= cold-value (clojure.core/deref backend)))
  (error? [_this]
    (instance? #?(:clj Throwable :cljs js/Error)
               (clojure.core/deref backend)))

  IDeref
  (#?(:clj deref :cljs -deref)
    [this]
    ((fsafe *tracer*) :deref this)
    (if (cold? this)
      (throw (SignalColdException. {:signal this}))
      (let [value (clojure.core/deref backend)]
        (cond-> value (instance? #?(:clj Throwable :cljs js/Error) value) throw))))

  IAlter
  (-alter!
    [_this fun args]
    (let [result (p/promise)
          mutation-fn (fn [old-value]
                        (try
                          (apply fun old-value args)
                          (catch #?(:clj Throwable :cljs :default) e e)))]
      (with-lock [alter-lock]
        (when (nil? @alter-queue)
          (reset! alter-queue
                  (let [q (f/flow {:buffer buffer-spec})]
                    (f/consume (fn [[mutation-fn result]]
                                 (let [new-value (swap! backend mutation-fn)]
                                   (p/resolve! result new-value)
                                   (when (= new-value cold-value)
                                     (with-lock [alter-lock]
                                       (f/close! q)
                                       (reset! alter-queue nil))))) q)
                    q)))
        (f/put! @alter-queue [mutation-fn result]))
      result))

  #?(:clj IRef :cljs IWatchable)
  (#?(:clj addWatch :cljs -add-watch)
    [this watch-key watch-fn]
    (let [watch-fn (fn [old-value new-value]
                     (try
                       (watch-fn watch-key this old-value new-value)
                       (catch #?(:clj Throwable :cljs :default) _)))]
      (add-watch
       backend watch-key
       (fn [_key _ref old-value new-value]
         (when (not= old-value new-value)
           (watch-fn old-value new-value))))
      (swap! watches assoc watch-key watch-fn)
      (when-not (cold? this)
        (watch-fn cold-value (clojure.core/deref backend))))
    this)
  (#?(:clj removeWatch :cljs -remove-watch)
    [this watch-key]
    (remove-watch backend watch-key)
    (swap! watches dissoc watch-key)
    this)

  #?(:clj IObj :cljs IWithMeta)
  (#?(:clj withMeta :cljs -with-meta)
    [_ meta-map]
    (Signal. backend watches alter-queue alter-lock buffer-spec meta-map))

  IMeta
  (#?(:clj meta :cljs -meta)
    [_]
    meta-map)

  #?@(:cljs
      [IPrintWithWriter
       (-pr-writer [this w _opts] (write-all w (pr-signal this)))])

  IWatchWatchers
  (add-watcher-watch [this watch-key watch-fn]
    (add-watch watches watch-key (fn [key _ref old-value new-value]
                                   (watch-fn key this old-value new-value))))
  (remove-watcher-watch [_this watch-key]
    (remove-watch watches watch-key)))

#?(:clj
   (defmethod print-method Signal
     [^Signal s w]
     (.write ^java.io.Writer w ^String (pr-signal s))))

#?(:clj
   (defmethod simple-dispatch Signal
     [^Signal s]
     (print (pr-signal s))))

(defn signal
  ([] (signal cold-value))
  ([state]
   (signal state [:fixed 1]))
  ([state buffer]
   (let [s (Signal. (atom state) (atom {}) (atom nil) #?(:clj (ReentrantLock.) :cljs nil) buffer nil)]
     ((fsafe *tracer*) :create s)
     s)))

(defn inspect
  [s]
  (let [k #?(:clj (UUID/randomUUID) :cljs (random-uuid))
        p (p/promise)]
    (add-watch s k (fn [_ _ _ v]
                     (p/resolve! p v)
                     (remove-watch s k)))
    p))

#?(:clj
   (defmacro with-tracer
     [tracer-fn & body]
     `(binding [*tracer* ~tracer-fn]
        ~@body)))


;;; Private

(defn- pr-signal
  [^Signal signal]
  (ust/format (str "#<signum/signal@" #?(:clj "0x%x" :cljs "%s") ": %s>") (hash signal)
              (try (pr-str #?(:clj @(.backend signal) :cljs @(j/get signal :backend)))
                   #?(:clj  (catch Throwable e (.getMessage e))
                      :cljs (catch js/Error e (j/get e :message))))))
