;;   Copyright (c) 7theta. All rights reserved.
;;   The use and distribution terms for this software are covered by the
;;   MIT License (https://opensource.org/licenses/MIT) which can also be
;;   found in the LICENSE file at the root of this distribution.
;;
;;   By using this software in any fashion, you are agreeing to be bound by
;;   the terms of this license.
;;   You must not remove this notice, or any others, from this software.

(ns schema.core
  (:require [malli.core :as m]
            [clojure.walk :refer [postwalk]]))

(declare parse-meta parse-args)

(defmacro >fn
  [& args]
  (let [{:keys [params in-schema out-schema body]} (parse-args args)
        in-schema (into [:tuple] in-schema)]
    `(let [in-schema# ~in-schema
           out-schema# ~out-schema
           valid-in# (m/validator in-schema#)
           valid-out# (m/validator out-schema#)
           explain-in# (m/explainer in-schema#)
           explain-out# (m/explainer out-schema#)]
       (fn ~params
         (when-not (valid-in# ~params)
           (throw (ex-info "Arguments do not conform to schema"
                           (explain-in# ~params))))
         (let [result# (do ~@body)]
           (if (valid-out# result#)
             result#
             (throw (ex-info "Result does not conform to schema"
                             (explain-out# result#)))))))))

(defmacro >defn
  [name & args]
  (let [{:keys [meta-map params]} (parse-meta args)]
    `(def ~(with-meta name meta-map)
       (>fn ~@params))))

(defmacro >defn-
  [name & args]
  (let [{:keys [meta-map params]} (parse-meta args)]
    `(def ~(with-meta name (assoc meta-map :private true))
       (>fn ~@params))))


;;; Private

(defn- parse-meta
  [args]
  (let [meta-map (if (string? (first args)) {:doc (first args)} {})
        args (if (string? (first args)) (next args) args)

        meta-map (if (map? (first args)) (conj meta-map (first args)) meta-map)
        args (if (map? (first args)) (next args) args)

        params (first args)
        meta-map (assoc meta-map :arglists `(quote [~params]))]
    {:meta-map meta-map
     :params args}))

(defn- parse-args
  [args]
  (let [params (first args)
        args (next args)
        [ins _ [outs]] (->> args
                            first
                            (postwalk (fn [x] (if (= '_ x) 'any? x)))
                            (partition-by (partial = '=>)))
        body (next args)]
    {:in-schema (vec ins)
     :out-schema outs
     :params params
     :body body}))
