(ns dev
  "Tools for interactive development with the REPL. This file should
  not be included in a production build of the application."
  (:require [via.endpoint :as via-e]
            [vectio.http :as http]
            [vectio.middleware.gzip :as gzip]
            [fluxus.flow :as f]
            [spectator.log :refer [error]]
            [utilis.jar :as jar]
            [reitit.ring :refer [ring-handler router routes]]
            [clojure.java.io :as io]

            [integrant.core :as ig]
            [integrant.repl :refer [clear go halt init reset reset-all]]
            [integrant.repl.state :refer [system]]

            [clojure.tools.namespace.repl :refer [refresh refresh-all disable-reload!]]
            [clojure.repl :refer [apropos dir doc find-doc pst source]]
            [clojure.test :refer [run-tests run-all-tests]]
            [clojure.pprint :refer [pprint]]
            [clojure.reflect :refer [reflect]]

            [signum.subs :refer [subscribe]]))

(disable-reload! (find-ns 'integrant.core))

(f/set-default-on-error! (fn [s e] (error [:fluxus.flow/error s] e)))

(defn via-handler
  [via-endpoint request]
  (when-let [{:keys [stream response]} (http/websocket-stream-response request)]
    (via-e/register via-endpoint stream {:coeffects {:request request}})
    response))

(defmethod ig/init-key :sarcina.dev/ring-handler
  [_ {:keys [apk-server via-endpoint]}]
  (ring-handler
   (router
    [["/via" #(via-handler via-endpoint %)]
     ["/packages/apk/:id/:version-code/:asset"
      {:get {:handler (:get-handler apk-server)}}]])
   (routes)
   {:middleware [gzip/middleware]}))

(def dev-config
  {:via/endpoint
   {:apk-server (ig/ref :sarcina/apk)
    :exports {:namespaces #{:sarcina/apk}}}

   :via/subs
   {:endpoint (ig/ref :via/endpoint)}

   :sarcina/apk
   {:url-prefix "/packages/apk"
    :apk-dir "packages/apks"
    :logger (ig/ref :spectator/log)}

   :sarcina.dev/ring-handler
   {:via-endpoint (ig/ref :via/endpoint)
    :apk-server (ig/ref :sarcina/apk)}

   :vectio.http/server
   {:host "0.0.0.0"
    :port 3449
    :tls {:local {:cert (io/file "dev-resources/certs/server.crt")
                  :key (io/file "dev-resources/certs/server.key")}}
    :ring-handler (ig/ref :sarcina.dev/ring-handler)}

   :spectator/log
   {:level :debug
    :logger (ig/ref :spectator.logger/console)
    :service-name (str "com.7theta/" "sarcina")
    :service-version (jar/version "com.7theta" "sarcina")}

   :spectator.logger/console {}})

(ig/load-namespaces dev-config)

(integrant.repl/set-prep! (constantly dev-config))

(defn inspect-apk-packages
  []
  (let [s (subscribe [:sarcina.apk/packages])
        k ::inspect-apk-packages
        p (promise)]
    (add-watch s k (fn [_ _ _ v]
                     (deliver p v)
                     (remove-watch s k)))
    p))
