(ns sarcina.apk
  (:require [sarcina.schema :refer [APK]]
            [via-schema.core :refer [>fn]]
            [signum.core :refer [signal alter! reg-sub]]
            [fs-watch.core :as fs]
            [utilis.fs :refer [ls with-temp mkdir]]
            [utilis.map :refer [map-vals]]
            [utilis.types.number :refer [string->long]]
            [clojure.java.io :as io]
            [clojure.string :refer [ends-with?]]
            [spectator.log :refer [error info debug]]
            [integrant.core :as ig])
  (:import [net.dongliu.apk.parser ApkFile]
           [java.util.zip GZIPInputStream]))

(def mime-type "application/vnd.android.package-archive")

(defonce ^:private apk-cache (signal nil))
(defonce ^:private url-prefix (atom nil))

(declare reset-cache! get-handler)

(defmethod ig/init-key :sarcina/apk
  [_ {:keys [url-prefix apk-dir] :as opts}]
  (reset! sarcina.apk/url-prefix url-prefix)
  (mkdir apk-dir :recursive true)
  (let [apk-dir (io/file apk-dir)]
    (reset-cache! apk-dir)
    {:watcher (fs/watch
               (fn [_] (reset-cache! apk-dir))
               [(.getAbsolutePath apk-dir)])
     :url-prefix url-prefix
     :get-handler get-handler}))

(defmethod ig/halt-key! :sarcina/apk
  [_ {:keys [watcher]}]
  (reset! sarcina.apk/url-prefix nil)
  (fs/stop watcher))

(reg-sub
 :sarcina.apk/packages
 (>fn [_]
   [[:tuple _] => [:map-of :string [:map-of :int APK]]]
   (map-vals
    (partial map-vals
       (fn [apk]
         (-> (select-keys apk [:id :display-name :version :permissions])
             (assoc :urls (let [prefix (str @url-prefix "/" (:id apk) "/" (-> apk :version :code) "/")]
                            {:apk (str prefix "app.apk")
                             :icon (str prefix "icon.png")})))))
    @apk-cache)))


;;; Private

(defn- get-handler
  [{{:keys [id version-code asset]} :path-params :as request}]
  (debug [:sarcina/apk :get id version-code asset])
  (try
    (if-let [version-code (string->long version-code)]
      (case asset
        "icon.png"
        {:status 200
         :headers {"Content-Type" "image/png"}
         :body (get-in @apk-cache [id version-code :icon])}
        "app.apk"
        (let [file (io/file (get-in @apk-cache [id version-code :path]))]
          (if (.exists file)
            {:status 200
             :headers {"content-type" mime-type
                       "content-encoding" "gzip"}
             :body file}
            {:status 404}))
        {:status 400
         :body "invalid asset"})
      {:status 400
       :body "invalid version-code"})
    (catch Exception e
      (error [:sarcina/apk :get-handler request] e)
      {:status 500})))

(defn- apk-metadata
  [apk]
  (with-temp [apk-file]
    (with-open [gz (GZIPInputStream. (io/input-stream apk))]
      (io/copy gz apk-file))
    (let [file (ApkFile. apk-file)
          metadata (.getApkMeta file)]
      {:id (.getPackageName metadata)
       :display-name (.getLabel metadata)
       :path (io/file apk)
       :version {:name (.getVersionName metadata)
                 :code (.getVersionCode metadata)}
       :permissions {:declared (mapv #(.getName %) (.getPermissions metadata))
                     :uses (vec (.getUsesPermissions metadata))}
       :icon (->> (.getAllIcons file)
                  (map #(.getData %))
                  (sort-by alength)
                  last)})))

(defn- reset-cache!
  [apk-dir]
  (locking apk-cache
    (let [files (ls apk-dir)
          cache (reduce
                 (fn [cache file]
                   (try
                     (let [path (.getAbsolutePath file)]
                       (if (ends-with? path ".apk.gz")
                         (let [{:keys [id version] :as apk} (apk-metadata (.getAbsolutePath file))]
                           (debug [:sarcina/apk :found apk])
                           (assoc-in cache [id (:code version)] apk))
                         (do
                           (info [:sarcina/apk :ignore path])
                           cache)))
                     (catch Exception e
                       (error [:sarcina/apk :parse-error file] e)
                       cache)))
                 {} files)]
      (alter! apk-cache (constantly cache)))))
